-- Roblox Electrode Plugin
-- This plugin integrates with the AI backend to generate and paste Lua code

local TweenService = game:GetService("TweenService")
local ChangeHistoryService = game:GetService("ChangeHistoryService")
local SelectionService = game:GetService("Selection")
local TextService = game:GetService("TextService")
local RunService = game:GetService("RunService")
local HttpService = game:GetService("HttpService")
local StudioService = game:GetService("StudioService")
local CollectionService = game:GetService("CollectionService")
local LogService = game:GetService("LogService")
local Network = {}

local ThoughtConfig = {
    stages = {
        { id = "request", label = "Understanding request" },
        { id = "snapshot", label = "Capturing workspace snapshot" },
        { id = "model", label = "Drafting strategy" },
        { id = "response", label = "Preparing response" },
        { id = "apply", label = "Applying changes" }
    },
    colors = {
        idle = Color3.fromRGB(189, 195, 199),
        active = Color3.fromRGB(52, 152, 219),
        done = Color3.fromRGB(46, 204, 113)
    },
    stageProgress = {}
}
local THOUGHT_WRAPPER_ORDER, PROGRESS_DRIFT_CAP = 1000000, 0.92

local STATUS_COLORS = {
    success = Color3.fromRGB(135, 235, 185),
    warning = Color3.fromRGB(255, 210, 120),
    error = Color3.fromRGB(255, 150, 160),
    info = Color3.fromRGB(150, 180, 255)
}

local Style = {}

do
    local stageProgress = ThoughtConfig.stageProgress
    local stageCount = math.max(#ThoughtConfig.stages, 1)
    for index = 1, stageCount do
        local normalized = index / stageCount
        local eased = math.clamp(0.08 + normalized * 0.82, 0, 0.98)
        stageProgress[index] = eased
    end
    stageProgress[stageCount] = 1
end

local NarrationConfig = {
    minInterval = 15,
    maxInterval = 25,
    initialDelay = 4,
    beats = {
        "🔍 Taking a look at what you've got here...",
        "🗺️ Figuring out how to put this together...",
        "📋 Checking what scripts you already have...",
        "🔌 Making sure I understand your setup...",
        "📁 Deciding where everything should go...",
        "💭 Working through the logic...",
        "🎨 Keeping it in your style...",
        "✅ Double-checking the structure...",
        "📝 Writing it all out...",
        "🔗 Connecting the pieces...",
        "🔍 Making sure nothing's missing...",
        "⚡ Keeping it fast and smooth...",
        "🛡️ Keeping it safe and compliant...",
        "🎯 Matching your project's vibe...",
        "✨ Almost done here...",
        "🎨 Adding the final touches...",
        "🔎 Making sure it all fits together...",
        "🏗️ Building it out...",
        "✔️ Testing the approach...",
        "🎉 Just about ready!"
    }
}

local showThoughtStages -- forward declaration
local hideThoughtStages -- forward declaration
local setThoughtStage -- forward declaration
local showTypingIndicator -- forward declaration
local appendConversationMessage -- forward declaration
local scriptAnalysis = {} -- Initialize early to avoid nil errors

-- Consolidated nil variables and simple values
local currentThoughtStageIndex, thoughtProgressTween, progressDriftConnection, loadingIconTween, progressGradientTween = nil, nil, nil, nil, nil
local currentTheme, currentModel = "light", "gpt"

local LIGHT_THEME = {
    background = Color3.fromRGB(255, 255, 255),
    text = Color3.fromRGB(0, 0, 0),
    inputBackground = Color3.fromRGB(248, 249, 250),
    inputBorder = Color3.fromRGB(233, 236, 239),
    cardBackground = Color3.fromRGB(248, 249, 252),
    cardStroke = Color3.fromRGB(225, 229, 235),
    subtext = Color3.fromRGB(104, 113, 122)
}

local DARK_THEME = {
    background = Color3.fromRGB(40, 40, 40),
    text = Color3.fromRGB(255, 255, 255),
    inputBackground = Color3.fromRGB(60, 60, 60),
    inputBorder = Color3.fromRGB(80, 80, 80),
    cardBackground = Color3.fromRGB(55, 55, 60),
    cardStroke = Color3.fromRGB(85, 85, 95),
    subtext = Color3.fromRGB(185, 190, 200)
}

local themes = {
    light = LIGHT_THEME,
    dark = DARK_THEME
}

function Style.ensureChild(parent, className, name)
    local child = parent:FindFirstChild(name)
    if not child then
        child = Instance.new(className)
        child.Name = name
        child.Parent = parent
    end
    return child
end

function Style.applyGlassEffect(frame, options)
    options = options or {}
    frame.BackgroundTransparency = options.transparency or 0.2
    frame.BackgroundColor3 = options.baseColor or Color3.fromRGB(16, 18, 40)
    frame.BorderSizePixel = 0

    local corner = Style.ensureChild(frame, "UICorner", "GlassCorner")
    corner.CornerRadius = UDim.new(0, options.cornerRadius or 14)

    local stroke = Style.ensureChild(frame, "UIStroke", "GlassStroke")
    stroke.Thickness = options.strokeThickness or 1.5
    stroke.Color = options.strokeColor or Color3.fromRGB(255, 255, 255)
    stroke.Transparency = options.strokeTransparency or 0.6
    stroke.ApplyStrokeMode = Enum.ApplyStrokeMode.Border

    local gradient = Style.ensureChild(frame, "UIGradient", "GlassGradient")
    gradient.Color = options.gradient or ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(255, 255, 255)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(180, 190, 255))
    })
    gradient.Rotation = options.gradientRotation or 30
    gradient.Transparency = options.gradientTransparency or NumberSequence.new({
        NumberSequenceKeypoint.new(0, 0.2),
        NumberSequenceKeypoint.new(1, 0.45)
    })
end

function Style.applyGlow(parent, name, color, size, position)
    local glow = Style.ensureChild(parent, "ImageLabel", name)
    glow.BackgroundTransparency = 1
    glow.Image = "rbxassetid://4691707148"
    glow.ImageColor3 = color
    glow.ImageTransparency = 0.35
    glow.Size = size or UDim2.new(0, 320, 0, 320)
    glow.Position = position or UDim2.new(-0.1, 0, -0.12, 0)
    glow.ZIndex = -10
    glow.ScaleType = Enum.ScaleType.Fit
end

function Style.stylePrimaryButton(button)
    button.AutoButtonColor = false
    button.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
    button.BackgroundTransparency = 0.05
    button.TextColor3 = Color3.fromRGB(255, 255, 255)
    button.Font = Enum.Font.SourceSansSemibold

    local corner = Style.ensureChild(button, "UICorner", "ButtonCorner")
    corner.CornerRadius = UDim.new(0, 10)

    local gradient = Style.ensureChild(button, "UIGradient", "ButtonGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(160, 130, 255)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(80, 110, 255))
    })
    gradient.Rotation = 90
end

function Style.styleSecondaryButton(button)
    button.AutoButtonColor = false
    button.BackgroundColor3 = Color3.fromRGB(24, 28, 60)
    button.BackgroundTransparency = 0.2
    button.TextColor3 = Color3.fromRGB(255, 255, 255)
    button.Font = Enum.Font.SourceSansSemibold

    local corner = Style.ensureChild(button, "UICorner", "ButtonCorner")
    corner.CornerRadius = UDim.new(0, 10)

    local gradient = Style.ensureChild(button, "UIGradient", "ButtonGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(46, 55, 110)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(28, 32, 72))
    })
    gradient.Rotation = 90
    gradient.Transparency = NumberSequence.new({
        NumberSequenceKeypoint.new(0, 0.1),
        NumberSequenceKeypoint.new(1, 0.35)
    })

    local stroke = Style.ensureChild(button, "UIStroke", "ButtonStroke")
    stroke.Thickness = 1
    stroke.Color = Color3.fromRGB(120, 150, 255)
    stroke.Transparency = 0.4
end

function Style.setButtonGradient(button, colorA, colorB, transparencySequence)
    local gradient = Style.ensureChild(button, "UIGradient", "ButtonGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, colorA),
        ColorSequenceKeypoint.new(1, colorB)
    })
    if transparencySequence then
        gradient.Transparency = transparencySequence
    end
end

function Style.styleInputBox(textbox)
    textbox.BackgroundColor3 = Color3.fromRGB(20, 24, 58)
    textbox.BackgroundTransparency = 0.22
    textbox.BorderSizePixel = 0
    textbox.TextColor3 = Color3.fromRGB(255, 255, 255)
    textbox.PlaceholderColor3 = Color3.fromRGB(150, 160, 190)
    textbox.Font = Enum.Font.Gotham

    local corner = Style.ensureChild(textbox, "UICorner", "InputCorner")
    corner.CornerRadius = UDim.new(0, 12)

    local stroke = Style.ensureChild(textbox, "UIStroke", "InputStroke")
    stroke.Thickness = 1
    stroke.Color = Color3.fromRGB(120, 150, 255)
    stroke.Transparency = 0.45

    local gradient = Style.ensureChild(textbox, "UIGradient", "InputGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(36, 42, 88)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(24, 28, 66))
    })
    gradient.Rotation = 110
    gradient.Transparency = NumberSequence.new({
        NumberSequenceKeypoint.new(0, 0.1),
        NumberSequenceKeypoint.new(1, 0.4)
    })
end

-- Attach to themes table to avoid module-level local
themes.getActive = function()
    local palette = themes
    if palette then
        local selected = currentTheme and palette[currentTheme]
        if selected then
            return selected
        end
        if palette.light then
            return palette.light
        end
    end
    return LIGHT_THEME
end
local getActiveTheme = themes.getActive

-- Attach to Network table to avoid module-level local
Network.formatError = function(err)
    if err == nil then
        return "unknown error"
    end

    if typeof(err) == "string" then
        return err
    end

    if typeof(err) == "table" then
        local detail = err.detail or err.error or err.message or err.reason or err.description
        if detail then
            return tostring(detail)
        end

        local success, encoded = pcall(HttpService.JSONEncode, HttpService, err)
        if success then
            return encoded
        end
    end

    return tostring(err)
end

pcall(function()
    -- Increase timeout to maximum allowed (300 seconds) for Opus and complex requests
    -- Note: Roblox may have a hard limit around 120s, but we set it as high as possible
    if HttpService.HttpTimeout and HttpService.HttpTimeout < 300 then
        HttpService.HttpTimeout = 300
    end
end)

local Plugin = script.Parent
local PluginButton = plugin:CreateToolbar("Electrode"):CreateButton(
    "Generate Code",
    "Generate Lua code with AI",
    "rbxassetid://0" -- You'll need to upload an icon
)

local Widget = plugin:CreateDockWidgetPluginGui(
    "AICodingAssistantWidget",
    DockWidgetPluginGuiInfo.new(
        Enum.InitialDockState.Float,
        false,
        false,
        400,
        300,
        200,
        150
    )
)
Widget.ZIndexBehavior = Enum.ZIndexBehavior.Sibling

-- Create the main frame
local MainFrame = Widget:FindFirstChild("Frame")
if not MainFrame then
    MainFrame = Instance.new("Frame")
    MainFrame.Name = "Frame"
MainFrame.Parent = Widget
else
    MainFrame.Name = "Frame"
end
MainFrame.Size = UDim2.new(1, 0, 1, 0)
MainFrame.BackgroundColor3 = Color3.fromRGB(6, 6, 26)
MainFrame.BorderSizePixel = 0
MainFrame.ClipsDescendants = false

local backgroundGradient = Style.ensureChild(MainFrame, "UIGradient", "BackgroundGradient")
backgroundGradient.Color = ColorSequence.new({
    ColorSequenceKeypoint.new(0, Color3.fromRGB(18, 22, 68)),
    ColorSequenceKeypoint.new(1, Color3.fromRGB(6, 6, 26))
})
backgroundGradient.Rotation = 65
backgroundGradient.Transparency = NumberSequence.new({
    NumberSequenceKeypoint.new(0, 0),
    NumberSequenceKeypoint.new(1, 0.25)
})

Style.applyGlow(MainFrame, "TopGlow", Color3.fromRGB(110, 98, 255), UDim2.new(0, 380, 0, 380), UDim2.new(-0.15, 0, -0.18, 0))
Style.applyGlow(MainFrame, "BottomGlow", Color3.fromRGB(161, 92, 255), UDim2.new(0, 420, 0, 420), UDim2.new(0.65, 0, 0.65, 0))

local existingChildren = MainFrame:GetChildren()

-- Primary layout containers
local TitleHeight = 40
local TabsHeight = 38

local TopFrame = MainFrame:FindFirstChild("TopFrame")
if not TopFrame then
    TopFrame = Instance.new("Frame")
    TopFrame.Name = "TopFrame"
    TopFrame.Parent = MainFrame
end
TopFrame.BackgroundTransparency = 1
TopFrame.Size = UDim2.new(1, 0, 0, TitleHeight + TabsHeight)

for _, child in ipairs(existingChildren) do
    if child ~= TopFrame then
        child:Destroy()
    end
end

for _, child in ipairs(TopFrame:GetChildren()) do
    child:Destroy()
end

_G.MainFrame = MainFrame
_G.TopFrame = TopFrame

local TabsContainer = Instance.new("Frame")
TabsContainer.Name = "TabsContainer"
TabsContainer.Size = UDim2.new(1, -20, 0, TabsHeight)
TabsContainer.Position = UDim2.new(0, 10, 0, TitleHeight + 6)
TabsContainer.AnchorPoint = Vector2.new(0, 0)
TabsContainer.BorderSizePixel = 0
TabsContainer.Parent = TopFrame
Style.applyGlassEffect(TabsContainer, {
    transparency = 0.35,
    baseColor = Color3.fromRGB(18, 22, 58),
    cornerRadius = 12,
    gradientRotation = 90,
    gradientTransparency = NumberSequence.new({
        NumberSequenceKeypoint.new(0, 0.2),
        NumberSequenceKeypoint.new(1, 0.55)
    })
})

local TabsLayout = Instance.new("UIListLayout")
TabsLayout.FillDirection = Enum.FillDirection.Horizontal
TabsLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
TabsLayout.VerticalAlignment = Enum.VerticalAlignment.Center
TabsLayout.Padding = UDim.new(0, 4)
TabsLayout.SortOrder = Enum.SortOrder.LayoutOrder
TabsLayout.Parent = TabsContainer

local function createTabButton(name, order)
    local button = Instance.new("TextButton")
    button.Name = name .. "TabButton"
    button.Size = UDim2.new(0, 140, 0, 28)
    button.BorderSizePixel = 0
    button.AutoButtonColor = false
    button.Text = name
    button.TextColor3 = Color3.fromRGB(255, 255, 255)
    button.Font = Enum.Font.SourceSansSemibold
    button.TextSize = 13
    button.LayoutOrder = order
    button.Parent = TabsContainer

    local corner = Style.ensureChild(button, "UICorner", "TabCorner")
    corner.CornerRadius = UDim.new(0, 10)

    local gradient = Style.ensureChild(button, "UIGradient", "TabGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(65, 80, 180)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(40, 50, 130))
    })
    gradient.Rotation = 90

    local stroke = Style.ensureChild(button, "UIStroke", "TabStroke")
    stroke.Thickness = 1
    stroke.Color = Color3.fromRGB(140, 150, 230)
    stroke.Transparency = 0.4

    return button
end

local SettingsTabButton = createTabButton("Settings", 1)
local AgentTabButton = createTabButton("Agent", 2)
local VFXTabButton = createTabButton("Library", 3)

local ContentFrame = Instance.new("Frame")
ContentFrame.Name = "ContentFrame"
ContentFrame.Size = UDim2.new(1, 0, 1, -(TitleHeight + TabsHeight + 10))
ContentFrame.Position = UDim2.new(0, 0, 0, TitleHeight + TabsHeight + 10)
ContentFrame.BackgroundTransparency = 1
ContentFrame.Parent = MainFrame

_G.ContentFrame = ContentFrame

local SettingsPage = Instance.new("Frame")
SettingsPage.Name = "SettingsPage"
SettingsPage.BackgroundTransparency = 1
SettingsPage.Size = UDim2.new(1, 0, 1, 0)
SettingsPage.Parent = ContentFrame

local AgentPage = Instance.new("Frame")
AgentPage.Name = "AgentPage"
AgentPage.BackgroundTransparency = 1
AgentPage.Size = UDim2.new(1, 0, 1, 0)
AgentPage.Visible = false
AgentPage.Parent = ContentFrame

local VFXPage = Instance.new("Frame")
VFXPage.Name = "VFXPage"
VFXPage.BackgroundTransparency = 1
VFXPage.Size = UDim2.new(1, 0, 1, 0)
VFXPage.Visible = false
VFXPage.Parent = ContentFrame

_G.SettingsPage = SettingsPage
_G.AgentPage = AgentPage
_G.VFXPage = VFXPage

local SettingsUI, VFXUI, AgentUI, scriptCache = {}, {}, {}, {}

-- Agent UI components
AgentUI.Padding = Instance.new("UIPadding")
AgentUI.Padding.PaddingTop = UDim.new(0, 10)
AgentUI.Padding.PaddingBottom = UDim.new(0, 10)
AgentUI.Padding.PaddingLeft = UDim.new(0, 10)
AgentUI.Padding.PaddingRight = UDim.new(0, 10)
AgentUI.Padding.Parent = AgentPage

AgentUI.Layout = Instance.new("UIListLayout")
AgentUI.Layout.FillDirection = Enum.FillDirection.Vertical
AgentUI.Layout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.Layout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.Layout.Padding = UDim.new(0, 8)
AgentUI.Layout.Parent = AgentPage

AgentUI.Header = Instance.new("TextLabel")
AgentUI.Header.Name = "AgentHeader"
AgentUI.Header.BackgroundTransparency = 1
AgentUI.Header.Size = UDim2.new(1, 0, 0, 28)
AgentUI.Header.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.Header.Font = Enum.Font.GothamSemibold
AgentUI.Header.TextSize = 22
AgentUI.Header.TextColor3 = Color3.fromRGB(230, 235, 255)
AgentUI.Header.Text = "Conversational Agent"
AgentUI.Header.LayoutOrder = 1
AgentUI.Header.Parent = AgentPage

AgentUI.Controls = Instance.new("Frame")
AgentUI.Controls.Name = "AgentControls"
AgentUI.Controls.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.Controls.Size = UDim2.new(1, 0, 0, 48)
AgentUI.Controls.BackgroundTransparency = 1
AgentUI.Controls.LayoutOrder = 2
AgentUI.Controls.Parent = AgentPage

AgentUI.ControlsLayout = Instance.new("UIListLayout")
AgentUI.ControlsLayout.FillDirection = Enum.FillDirection.Horizontal
AgentUI.ControlsLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ControlsLayout.VerticalAlignment = Enum.VerticalAlignment.Top
AgentUI.ControlsLayout.Padding = UDim.new(0, 10)
AgentUI.ControlsLayout.Parent = AgentUI.Controls

AgentUI.ModelSelectorFrame = Instance.new("Frame")
AgentUI.ModelSelectorFrame.Name = "ModelSelectorFrame"
AgentUI.ModelSelectorFrame.Size = UDim2.new(0.45, 0, 0, 48)
AgentUI.ModelSelectorFrame.AutomaticSize = Enum.AutomaticSize.XY
AgentUI.ModelSelectorFrame.LayoutOrder = 1
AgentUI.ModelSelectorFrame.Parent = AgentUI.Controls
Style.applyGlassEffect(AgentUI.ModelSelectorFrame, {
    transparency = 0.22,
    baseColor = Color3.fromRGB(22, 26, 62),
    cornerRadius = 12,
    gradientRotation = 115
})

AgentUI.ModelSelectorPadding = Instance.new("UIPadding")
AgentUI.ModelSelectorPadding.PaddingTop = UDim.new(0, 8)
AgentUI.ModelSelectorPadding.PaddingBottom = UDim.new(0, 8)
AgentUI.ModelSelectorPadding.PaddingLeft = UDim.new(0, 10)
AgentUI.ModelSelectorPadding.PaddingRight = UDim.new(0, 10)
AgentUI.ModelSelectorPadding.Parent = AgentUI.ModelSelectorFrame

AgentUI.ModelSelectorLayout = Instance.new("UIListLayout")
AgentUI.ModelSelectorLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.ModelSelectorLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ModelSelectorLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.ModelSelectorLayout.Padding = UDim.new(0, 6)
AgentUI.ModelSelectorLayout.Parent = AgentUI.ModelSelectorFrame

AgentUI.ModelSelectorLabel = Instance.new("TextLabel")
AgentUI.ModelSelectorLabel.BackgroundTransparency = 1
AgentUI.ModelSelectorLabel.Size = UDim2.new(1, 0, 0, 16)
AgentUI.ModelSelectorLabel.Text = "Agent Model"
AgentUI.ModelSelectorLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.ModelSelectorLabel.Font = Enum.Font.GothamSemibold
AgentUI.ModelSelectorLabel.TextSize = 14
AgentUI.ModelSelectorLabel.TextColor3 = Color3.fromRGB(210, 220, 255)
AgentUI.ModelSelectorLabel.Parent = AgentUI.ModelSelectorFrame

AgentUI.ModelButtons = Instance.new("Frame")
AgentUI.ModelButtons.Name = "AgentModelButtons"
AgentUI.ModelButtons.Size = UDim2.new(1, 0, 0, 30)
AgentUI.ModelButtons.BackgroundTransparency = 1
AgentUI.ModelButtons.Parent = AgentUI.ModelSelectorFrame

AgentUI.ModelButtonsLayout = Instance.new("UIListLayout")
AgentUI.ModelButtonsLayout.FillDirection = Enum.FillDirection.Horizontal
AgentUI.ModelButtonsLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ModelButtonsLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.ModelButtonsLayout.Padding = UDim.new(0, 8)
AgentUI.ModelButtonsLayout.Parent = AgentUI.ModelButtons

local function createAgentModelButton(label)
    local button = Instance.new("TextButton")
    button.Size = UDim2.new(0, 130, 0, 28)
    button.AutoButtonColor = false
    button.Text = label
    button.Font = Enum.Font.GothamSemibold
    button.TextSize = 13
        Style.styleSecondaryButton(button)

    button.Parent = AgentUI.ModelButtons
    return button
end

AgentUI.ModelOption1 = createAgentModelButton("Planning")
AgentUI.ModelOption2 = createAgentModelButton("Agent")

-- Legacy aliases for shipping bundle compatibility
AgentModelOption1 = AgentUI.ModelOption1
AgentModelOption2 = AgentUI.ModelOption2

AgentUI.TrustContainer = Instance.new("Frame")
AgentUI.TrustContainer.Name = "TrustContainer"
AgentUI.TrustContainer.Size = UDim2.new(0.35, 0, 0, 48)
AgentUI.TrustContainer.AutomaticSize = Enum.AutomaticSize.XY
AgentUI.TrustContainer.LayoutOrder = 2
AgentUI.TrustContainer.Parent = AgentUI.Controls
Style.applyGlassEffect(AgentUI.TrustContainer, {
    transparency = 0.22,
    baseColor = Color3.fromRGB(22, 26, 62),
    cornerRadius = 12,
    gradientRotation = 120
})

AgentUI.TrustPadding = Instance.new("UIPadding")
AgentUI.TrustPadding.PaddingTop = UDim.new(0, 8)
AgentUI.TrustPadding.PaddingBottom = UDim.new(0, 8)
AgentUI.TrustPadding.PaddingLeft = UDim.new(0, 10)
AgentUI.TrustPadding.PaddingRight = UDim.new(0, 10)
AgentUI.TrustPadding.Parent = AgentUI.TrustContainer

AgentUI.TrustLayout = Instance.new("UIListLayout")
AgentUI.TrustLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.TrustLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.TrustLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.TrustLayout.Padding = UDim.new(0, 6)
AgentUI.TrustLayout.Parent = AgentUI.TrustContainer

AgentUI.TrustLabel = Instance.new("TextLabel")
AgentUI.TrustLabel.BackgroundTransparency = 1
AgentUI.TrustLabel.Size = UDim2.new(1, 0, 0, 16)
AgentUI.TrustLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.TrustLabel.Font = Enum.Font.GothamSemibold
AgentUI.TrustLabel.TextSize = 14
AgentUI.TrustLabel.TextColor3 = Color3.fromRGB(210, 220, 255)
AgentUI.TrustLabel.Text = "Trusted Mode"
AgentUI.TrustLabel.Parent = AgentUI.TrustContainer

AgentUI.TrustToggle = Instance.new("TextButton")
AgentUI.TrustToggle.Name = "TrustToggle"
AgentUI.TrustToggle.Size = UDim2.new(0, 150, 0, 28)
AgentUI.TrustToggle.Text = "Require Approval"
AgentUI.TrustToggle.Font = Enum.Font.GothamSemibold
AgentUI.TrustToggle.TextSize = 13
AgentUI.TrustToggle.AutoButtonColor = false
Style.styleSecondaryButton(AgentUI.TrustToggle)
AgentUI.TrustToggle.Parent = AgentUI.TrustContainer

AgentUI.ClearChatButton = Instance.new("TextButton")
AgentUI.ClearChatButton.Name = "ClearChatButton"
AgentUI.ClearChatButton.Size = UDim2.new(0, 100, 0, 48)
AgentUI.ClearChatButton.AutoButtonColor = false
AgentUI.ClearChatButton.Text = "Clear Chat"
AgentUI.ClearChatButton.Font = Enum.Font.GothamSemibold
AgentUI.ClearChatButton.TextSize = 13
AgentUI.ClearChatButton.TextColor3 = Color3.fromRGB(255, 255, 255)
AgentUI.ClearChatButton.LayoutOrder = 3
AgentUI.ClearChatButton.Parent = AgentUI.Controls
Style.applyGlassEffect(AgentUI.ClearChatButton, {
    transparency = 0.22,
    baseColor = Color3.fromRGB(22, 26, 62),
    cornerRadius = 12,
    gradientRotation = 120
})

AgentUI.ClearChatPadding = Instance.new("UIPadding")
AgentUI.ClearChatPadding.PaddingTop = UDim.new(0, 8)
AgentUI.ClearChatPadding.PaddingBottom = UDim.new(0, 8)
AgentUI.ClearChatPadding.PaddingLeft = UDim.new(0, 12)
AgentUI.ClearChatPadding.PaddingRight = UDim.new(0, 12)
AgentUI.ClearChatPadding.Parent = AgentUI.ClearChatButton

AgentUI.ClearChatButton.MouseEnter:Connect(function()
    AgentUI.ClearChatButton.TextColor3 = Color3.fromRGB(255, 200, 200)
end)

AgentUI.ClearChatButton.MouseLeave:Connect(function()
    AgentUI.ClearChatButton.TextColor3 = Color3.fromRGB(255, 255, 255)
end)

AgentUI.Workspace = Instance.new("Frame")
AgentUI.Workspace.Name = "AgentWorkspace"
AgentUI.Workspace.Size = UDim2.new(1, 0, 1, -120)
AgentUI.Workspace.BackgroundTransparency = 1
AgentUI.Workspace.LayoutOrder = 3
AgentUI.Workspace.Parent = AgentPage

AgentUI.WorkspaceLayout = Instance.new("UIListLayout")
AgentUI.WorkspaceLayout.FillDirection = Enum.FillDirection.Horizontal
AgentUI.WorkspaceLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.WorkspaceLayout.VerticalAlignment = Enum.VerticalAlignment.Top
AgentUI.WorkspaceLayout.Padding = UDim.new(0, 10)
AgentUI.WorkspaceLayout.Parent = AgentUI.Workspace

AgentUI.ChatFrame = Instance.new("Frame")
AgentUI.ChatFrame.Name = "ChatFrame"
AgentUI.ChatFrame.Size = UDim2.new(0.65, -5, 1, 0)
AgentUI.ChatFrame.Parent = AgentUI.Workspace
Style.applyGlassEffect(AgentUI.ChatFrame, {
    transparency = 0.18,
    baseColor = Color3.fromRGB(20, 24, 58),
    cornerRadius = 14,
    gradientRotation = 90
})

AgentUI.ChatScroll = Instance.new("ScrollingFrame")
AgentUI.ChatScroll.Name = "ChatScroll"
AgentUI.ChatScroll.Size = UDim2.new(1, -20, 1, -120)
AgentUI.ChatScroll.Position = UDim2.new(0, 10, 0, 10)
AgentUI.ChatScroll.AutomaticCanvasSize = Enum.AutomaticSize.Y
AgentUI.ChatScroll.CanvasSize = UDim2.new(0, 0, 0, 0)
AgentUI.ChatScroll.BackgroundTransparency = 1
AgentUI.ChatScroll.BorderSizePixel = 0
AgentUI.ChatScroll.ScrollBarThickness = 8
AgentUI.ChatScroll.ScrollBarImageColor3 = Color3.fromRGB(102, 126, 234)
AgentUI.ChatScroll.ZIndex = 1
AgentUI.ChatScroll.Parent = AgentUI.ChatFrame

AgentUI.ChatScrollLayout = Instance.new("UIListLayout")
AgentUI.ChatScrollLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.ChatScrollLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ChatScrollLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.ChatScrollLayout.Padding = UDim.new(0, 6)
AgentUI.ChatScrollLayout.Parent = AgentUI.ChatScroll

local thoughtContainerHeight = 120
AgentUI.ThoughtWrapper = Instance.new("Frame")
AgentUI.ThoughtWrapper.Name = "ThoughtWrapper"
AgentUI.ThoughtWrapper.Size = UDim2.new(1, 0, 0, thoughtContainerHeight + 12)
AgentUI.ThoughtWrapper.BackgroundTransparency = 1
AgentUI.ThoughtWrapper.Visible = false
AgentUI.ThoughtWrapper.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.ThoughtWrapper.Parent = nil

local thoughtWrapperPadding = Instance.new("UIPadding")
thoughtWrapperPadding.PaddingLeft = UDim.new(0, 6)
thoughtWrapperPadding.PaddingRight = UDim.new(0, 6)
thoughtWrapperPadding.Parent = AgentUI.ThoughtWrapper
AgentUI.ThoughtWrapper.LayoutOrder = THOUGHT_WRAPPER_ORDER

AgentUI.ThoughtContainer = Instance.new("Frame")
AgentUI.ThoughtContainer.Name = "ThoughtContainer"
AgentUI.ThoughtContainer.Size = UDim2.new(1, 0, 0, thoughtContainerHeight)
AgentUI.ThoughtContainer.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.ThoughtContainer.BorderSizePixel = 0
AgentUI.ThoughtContainer.Visible = true
AgentUI.ThoughtContainer.ZIndex = 5
AgentUI.ThoughtContainer.ClipsDescendants = false
AgentUI.ThoughtContainer.Parent = AgentUI.ThoughtWrapper
Style.applyGlassEffect(AgentUI.ThoughtContainer, {
    transparency = 0.18,
    baseColor = Color3.fromRGB(24, 28, 66),
    cornerRadius = 12,
    gradientRotation = 115
})

AgentUI.ThoughtPadding = Instance.new("UIPadding")
AgentUI.ThoughtPadding.PaddingTop = UDim.new(0, 10)
AgentUI.ThoughtPadding.PaddingBottom = UDim.new(0, 12)
AgentUI.ThoughtPadding.PaddingLeft = UDim.new(0, 12)
AgentUI.ThoughtPadding.PaddingRight = UDim.new(0, 12)
AgentUI.ThoughtPadding.Parent = AgentUI.ThoughtContainer

AgentUI.ThoughtHeader = Instance.new("Frame")
AgentUI.ThoughtHeader.Name = "ThoughtHeader"
AgentUI.ThoughtHeader.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.ThoughtHeader.Size = UDim2.new(1, 0, 0, 32)
AgentUI.ThoughtHeader.BackgroundTransparency = 1
AgentUI.ThoughtHeader.Parent = AgentUI.ThoughtContainer

local thoughtHeaderLayout = Instance.new("UIListLayout")
thoughtHeaderLayout.FillDirection = Enum.FillDirection.Horizontal
thoughtHeaderLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
thoughtHeaderLayout.VerticalAlignment = Enum.VerticalAlignment.Center
thoughtHeaderLayout.Padding = UDim.new(0, 10)
thoughtHeaderLayout.Parent = AgentUI.ThoughtHeader

AgentUI.LoadingIcon = Instance.new("ImageLabel")
AgentUI.LoadingIcon.Name = "LoadingIcon"
AgentUI.LoadingIcon.Size = UDim2.new(0, 28, 0, 28)
AgentUI.LoadingIcon.BackgroundTransparency = 1
AgentUI.LoadingIcon.Image = "rbxassetid://10723384879"
AgentUI.LoadingIcon.ImageColor3 = Color3.fromRGB(140, 200, 255)
AgentUI.LoadingIcon.ImageTransparency = 0.05
AgentUI.LoadingIcon.ZIndex = 7
AgentUI.LoadingIcon.Visible = false
AgentUI.LoadingIcon.Parent = AgentUI.ThoughtHeader

AgentUI.ThoughtLabel = Instance.new("TextLabel")
AgentUI.ThoughtLabel.Name = "ThoughtLabel"
AgentUI.ThoughtLabel.BackgroundTransparency = 1
AgentUI.ThoughtLabel.Size = UDim2.new(1, 0, 0, 28)
AgentUI.ThoughtLabel.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.ThoughtLabel.Font = Enum.Font.GothamSemibold
AgentUI.ThoughtLabel.TextSize = 15
AgentUI.ThoughtLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.ThoughtLabel.TextColor3 = Color3.fromRGB(220, 230, 255)
AgentUI.ThoughtLabel.Text = "Electrode is building your request..."
AgentUI.ThoughtLabel.ZIndex = 6
AgentUI.ThoughtLabel.Parent = AgentUI.ThoughtHeader

AgentUI.ThoughtProgressTrack = Instance.new("Frame")
AgentUI.ThoughtProgressTrack.Name = "ThoughtProgressTrack"
AgentUI.ThoughtProgressTrack.Size = UDim2.new(1, 0, 0, 10)
AgentUI.ThoughtProgressTrack.Position = UDim2.new(0, 0, 0, 36)
AgentUI.ThoughtProgressTrack.BackgroundColor3 = Color3.fromRGB(18, 22, 52)
AgentUI.ThoughtProgressTrack.BackgroundTransparency = 0.1
AgentUI.ThoughtProgressTrack.ClipsDescendants = true
AgentUI.ThoughtProgressTrack.BorderSizePixel = 0
AgentUI.ThoughtProgressTrack.ZIndex = 6
AgentUI.ThoughtProgressTrack.Parent = AgentUI.ThoughtContainer

local progressCorner = Instance.new("UICorner")
progressCorner.CornerRadius = UDim.new(0, 5)
progressCorner.Parent = AgentUI.ThoughtProgressTrack

AgentUI.ThoughtProgressFill = Instance.new("Frame")
AgentUI.ThoughtProgressFill.Name = "ThoughtProgressFill"
AgentUI.ThoughtProgressFill.Size = UDim2.new(0, 0, 1, 0)
AgentUI.ThoughtProgressFill.AnchorPoint = Vector2.new(0, 0.5)
AgentUI.ThoughtProgressFill.Position = UDim2.new(0, 0, 0.5, 0)
AgentUI.ThoughtProgressFill.BackgroundColor3 = ThoughtConfig.colors.active
AgentUI.ThoughtProgressFill.BorderSizePixel = 0
AgentUI.ThoughtProgressFill.ZIndex = 7
AgentUI.ThoughtProgressFill.Parent = AgentUI.ThoughtProgressTrack

local progressFillCorner = Instance.new("UICorner")
progressFillCorner.CornerRadius = UDim.new(0, 5)
progressFillCorner.Parent = AgentUI.ThoughtProgressFill

AgentUI.ProgressGradient = Instance.new("UIGradient")
AgentUI.ProgressGradient.Color = ColorSequence.new({
    ColorSequenceKeypoint.new(0, Color3.fromRGB(120, 205, 255)),
    ColorSequenceKeypoint.new(0.5, Color3.fromRGB(164, 126, 255)),
    ColorSequenceKeypoint.new(1, Color3.fromRGB(255, 156, 222))
})
AgentUI.ProgressGradient.Rotation = 0
AgentUI.ProgressGradient.Offset = Vector2.new(0, 0)
AgentUI.ProgressGradient.Parent = AgentUI.ThoughtProgressFill

AgentUI.ThoughtWrapper.Parent = nil

-- Attach to Style table to avoid module-level local
Style.getStatusBadgeColors = function(status)
    if status == "done" then
        return Color3.fromRGB(214, 234, 223), Color3.fromRGB(39, 174, 96)
    elseif status == "running" then
        return Color3.fromRGB(214, 228, 244), Color3.fromRGB(52, 152, 219)
    else
        return Color3.fromRGB(241, 243, 247), Color3.fromRGB(87, 96, 111)
    end
end

local function applyPlanRowTheme(row, status)
    if not row then
        return
    end
    local theme = getActiveTheme()
    row.BackgroundColor3 = theme.cardBackground
    row.BackgroundTransparency = currentTheme == "dark" and 0.12 or 0
    local stroke = row:FindFirstChildWhichIsA("UIStroke")
    if stroke then
        stroke.Color = theme.cardStroke
    end
    local title = row:FindFirstChild("TitleLabel")
    if title then
        title.TextColor3 = theme.text
    end
    local detail = row:FindFirstChild("DetailLabel")
    if detail then
        detail.TextColor3 = theme.subtext
    end
    local statusBadge = row:FindFirstChild("StatusContainer", true)
    if statusBadge and statusBadge:IsA("TextLabel") then
        local bg, fg = Style.getStatusBadgeColors(status)
        statusBadge.BackgroundColor3 = bg
        statusBadge.TextColor3 = fg
        statusBadge.BackgroundTransparency = currentTheme == "dark" and 0.2 or 0
    end
end

local function stopLoadingIcon()
    if loadingIconTween then
        loadingIconTween:Cancel()
        loadingIconTween = nil
    end
    if AgentUI and AgentUI.LoadingIcon then
        AgentUI.LoadingIcon.Rotation = 0
    end
end

local function startLoadingIcon()
    if not (AgentUI and AgentUI.LoadingIcon) then
        return
    end
    stopLoadingIcon()
    AgentUI.LoadingIcon.Visible = true
    loadingIconTween = TweenService:Create(
        AgentUI.LoadingIcon,
        TweenInfo.new(1.2, Enum.EasingStyle.Linear, Enum.EasingDirection.InOut, -1),
        { Rotation = 360 }
    )
    loadingIconTween:Play()
end

local function stopProgressGradient()
    if progressGradientTween then
        progressGradientTween:Cancel()
        progressGradientTween = nil
    end
    if AgentUI and AgentUI.ProgressGradient then
        AgentUI.ProgressGradient.Rotation = 0
    end
end

local function startProgressGradient()
    if not (AgentUI and AgentUI.ProgressGradient) then
        return
    end
    stopProgressGradient()
    progressGradientTween = TweenService:Create(
        AgentUI.ProgressGradient,
        TweenInfo.new(3, Enum.EasingStyle.Linear, Enum.EasingDirection.InOut, -1),
        { Rotation = 360 }
    )
    progressGradientTween:Play()
end

local function stopProgressDrift()
    if progressDriftConnection then
        progressDriftConnection:Disconnect()
        progressDriftConnection = nil
    end
end

local function startProgressDrift()
    stopProgressDrift()
    if not (AgentUI and AgentUI.ThoughtProgressFill) then
        return
    end
    progressDriftConnection = RunService.Heartbeat:Connect(function(dt)
        local fill = AgentUI and AgentUI.ThoughtProgressFill
        if not fill then
            return
        end
        local current = fill.Size.X.Scale
        if current >= PROGRESS_DRIFT_CAP then
            return
        end
        local target = math.min(current + dt * 0.04, PROGRESS_DRIFT_CAP)
        if target > current then
            fill.Size = UDim2.new(target, 0, 1, 0)
        end
    end)
end

local function tweenProgressFill(targetRatio)
    if not (AgentUI and AgentUI.ThoughtProgressFill) then
        return
    end
    targetRatio = math.clamp(targetRatio, 0, 1)
    if thoughtProgressTween then
        thoughtProgressTween:Cancel()
        thoughtProgressTween = nil
    end
    thoughtProgressTween = TweenService:Create(
        AgentUI.ThoughtProgressFill,
        TweenInfo.new(1.2, Enum.EasingStyle.Quad, Enum.EasingDirection.Out),
        { Size = UDim2.new(targetRatio, 0, 1, 0) }
    )
    thoughtProgressTween:Play()
end

local function resetThoughtStages()
    stopProgressDrift()
    stopLoadingIcon()
    stopProgressGradient()
    currentThoughtStageIndex = nil
    if thoughtProgressTween then
        thoughtProgressTween:Cancel()
        thoughtProgressTween = nil
    end
    if AgentUI and AgentUI.ThoughtProgressFill then
        AgentUI.ThoughtProgressFill.Size = UDim2.new(0, 0, 1, 0)
    end
    if AgentUI and AgentUI.ThoughtLabel then
        AgentUI.ThoughtLabel.Text = "Electrode is building your request..."
    end
    if AgentUI and AgentUI.LoadingIcon then
        AgentUI.LoadingIcon.Visible = false
    end
end

setThoughtStage = function(stageIndex)
    if not (AgentUI and AgentUI.ThoughtProgressFill) then
        return
    end
    currentThoughtStageIndex = stageIndex
    local clampedIndex = math.clamp(stageIndex, 1, math.max(#ThoughtConfig.stages, 1))
    local target = ThoughtConfig.stageProgress[clampedIndex] or 1
    tweenProgressFill(target)
    if target >= 0.99 then
        stopProgressDrift()
    elseif not progressDriftConnection then
        startProgressDrift()
    end
end

showThoughtStages = function()
    if not AgentUI or not AgentUI.ThoughtContainer then
        return
    end
    resetThoughtStages()
    if AgentUI.ThoughtWrapper and AgentUI.ChatScroll then
        AgentUI.ThoughtWrapper.Parent = AgentUI.ChatScroll
        AgentUI.ThoughtWrapper.Visible = true
        AgentUI.ThoughtWrapper.LayoutOrder = THOUGHT_WRAPPER_ORDER
    end
    AgentUI.ThoughtContainer.Visible = true
    if AgentUI.ThoughtLabel then
        AgentUI.ThoughtLabel.TextColor3 = getActiveTheme().text
    end
    if AgentUI.ThoughtProgressTrack then
        AgentUI.ThoughtProgressTrack.BackgroundColor3 = getActiveTheme().inputBorder
    end
    startLoadingIcon()
    startProgressGradient()
    startProgressDrift()
    if type(updateChatCanvasSize) == "function" then
        updateChatCanvasSize()
    end
    setThoughtStage(1)
end

hideThoughtStages = function()
    if not AgentUI or not AgentUI.ThoughtContainer then
        return
    end
    resetThoughtStages()
    AgentUI.ThoughtContainer.Visible = false
    if AgentUI.ThoughtWrapper then
        AgentUI.ThoughtWrapper.Visible = false
        AgentUI.ThoughtWrapper.Parent = nil
    end
    if type(updateChatCanvasSize) == "function" then
        updateChatCanvasSize()
    end
end

showTypingIndicator = function(labelText)
    if not agentRequestInFlight then
        return
    end
    showThoughtStages()
    if AgentUI and AgentUI.ThoughtLabel then
        AgentUI.ThoughtLabel.Text = labelText or "Electrode is typing..."
    end
    if AgentUI and AgentUI.ThoughtProgressFill then
        AgentUI.ThoughtProgressFill.Size = UDim2.new(0, 0, 1, 0)
    end
    setThoughtStage(#ThoughtConfig.stages)
end

local function updateChatCanvasSize()
    -- With AutomaticCanvasSize.Y, we just need to auto-scroll to bottom when new messages arrive
    task.wait() -- Wait a frame for layout to update
    if AgentUI.ChatScroll then
        local maxScroll = math.max(0, AgentUI.ChatScroll.AbsoluteCanvasSize.Y - AgentUI.ChatScroll.AbsoluteSize.Y)
        AgentUI.ChatScroll.CanvasPosition = Vector2.new(0, maxScroll)
    end
end

-- Auto-scroll when content size changes (new messages added)
AgentUI.ChatScrollLayout:GetPropertyChangedSignal("AbsoluteContentSize"):Connect(updateChatCanvasSize)

AgentUI.ChatInputFrame = Instance.new("Frame")
AgentUI.ChatInputFrame.Name = "ChatInputFrame"
AgentUI.ChatInputFrame.Size = UDim2.new(1, -20, 0, 80)
AgentUI.ChatInputFrame.Position = UDim2.new(0, 10, 1, -90)
AgentUI.ChatInputFrame.BackgroundTransparency = 1
AgentUI.ChatInputFrame.ZIndex = 2
AgentUI.ChatInputFrame.Active = false
AgentUI.ChatInputFrame.ClipsDescendants = false
AgentUI.ChatInputFrame.Parent = AgentUI.ChatFrame

-- Create scrolling frame for the input box to allow infinite scrolling (same pattern as ChatScroll)
AgentUI.ChatInputScroll = Instance.new("ScrollingFrame")
AgentUI.ChatInputScroll.Name = "ChatInputScroll"
AgentUI.ChatInputScroll.Size = UDim2.new(1, -140, 0, 70)
AgentUI.ChatInputScroll.Position = UDim2.new(0, 0, 0, 0)
AgentUI.ChatInputScroll.AutomaticCanvasSize = Enum.AutomaticSize.Y
AgentUI.ChatInputScroll.CanvasSize = UDim2.new(0, 0, 0, 0)
AgentUI.ChatInputScroll.BackgroundTransparency = 1
AgentUI.ChatInputScroll.BorderSizePixel = 0
AgentUI.ChatInputScroll.ScrollBarThickness = 8
AgentUI.ChatInputScroll.ScrollBarImageColor3 = Color3.fromRGB(102, 126, 234)
AgentUI.ChatInputScroll.ZIndex = 3
AgentUI.ChatInputScroll.ClipsDescendants = true
AgentUI.ChatInputScroll.Parent = AgentUI.ChatInputFrame

AgentUI.ChatInputBox = Instance.new("TextBox")
AgentUI.ChatInputBox.Name = "ChatInputBox"
AgentUI.ChatInputBox.Size = UDim2.new(1, -10, 0, 70)
AgentUI.ChatInputBox.AutomaticSize = Enum.AutomaticSize.Y
AgentUI.ChatInputBox.Font = Enum.Font.Gotham
AgentUI.ChatInputBox.TextSize = 14
AgentUI.ChatInputBox.TextWrapped = true
AgentUI.ChatInputBox.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.ChatInputBox.TextYAlignment = Enum.TextYAlignment.Top
AgentUI.ChatInputBox.Text = ""
AgentUI.ChatInputBox.PlaceholderText = "Describe what you want electrode to help with... Markdown Supported"
AgentUI.ChatInputBox.ClearTextOnFocus = false
AgentUI.ChatInputBox.MultiLine = true
AgentUI.ChatInputBox.TextEditable = true
AgentUI.ChatInputBox.Active = true
AgentUI.ChatInputBox.Selectable = true
AgentUI.ChatInputBox.ClipsDescendants = false
AgentUI.ChatInputBox.ZIndex = 4
AgentUI.ChatInputBox.Parent = AgentUI.ChatInputScroll
Style.styleInputBox(AgentUI.ChatInputBox)

-- Add padding to the text box
local inputBoxPadding = Instance.new("UIPadding")
inputBoxPadding.PaddingTop = UDim.new(0, 5)
inputBoxPadding.PaddingBottom = UDim.new(0, 5)
inputBoxPadding.PaddingLeft = UDim.new(0, 5)
inputBoxPadding.PaddingRight = UDim.new(0, 5)
inputBoxPadding.Parent = AgentUI.ChatInputBox

-- Auto-scroll to bottom when text expands (similar to chat scroll)
AgentUI.ChatInputBox:GetPropertyChangedSignal("Text"):Connect(function()
    task.wait() -- Wait a frame for layout to update
    if AgentUI.ChatInputScroll and AgentUI.ChatInputBox then
        local maxScroll = math.max(0, AgentUI.ChatInputScroll.AbsoluteCanvasSize.Y - AgentUI.ChatInputScroll.AbsoluteSize.Y)
        AgentUI.ChatInputScroll.CanvasPosition = Vector2.new(0, maxScroll)
    end
end)

-- Also auto-scroll when canvas size changes
AgentUI.ChatInputScroll:GetPropertyChangedSignal("AbsoluteCanvasSize"):Connect(function()
    if AgentUI.ChatInputScroll then
        local maxScroll = math.max(0, AgentUI.ChatInputScroll.AbsoluteCanvasSize.Y - AgentUI.ChatInputScroll.AbsoluteSize.Y)
        AgentUI.ChatInputScroll.CanvasPosition = Vector2.new(0, maxScroll)
    end
end)

AgentUI.SendButton = Instance.new("TextButton")
AgentUI.SendButton.Name = "SendButton"
AgentUI.SendButton.Size = UDim2.new(0, 120, 0, 70)
AgentUI.SendButton.Position = UDim2.new(1, -120, 0, 0)
AgentUI.SendButton.Font = Enum.Font.GothamSemibold
AgentUI.SendButton.TextSize = 16
AgentUI.SendButton.Text = "Send"
AgentUI.SendButton.AutoButtonColor = false
AgentUI.SendButton.ZIndex = 3
AgentUI.SendButton.Parent = AgentUI.ChatInputFrame
Style.stylePrimaryButton(AgentUI.SendButton)

AgentUI.ContextFrame = Instance.new("Frame")
AgentUI.ContextFrame.Name = "ContextFrame"
AgentUI.ContextFrame.Size = UDim2.new(0.35, -5, 1, 0)
AgentUI.ContextFrame.BackgroundTransparency = 1
AgentUI.ContextFrame.BorderSizePixel = 0
AgentUI.ContextFrame.Parent = AgentUI.Workspace

AgentUI.ContextLayout = Instance.new("UIListLayout")
AgentUI.ContextLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.ContextLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ContextLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.ContextLayout.Padding = UDim.new(0, 12)
AgentUI.ContextLayout.Parent = AgentUI.ContextFrame

local function createContextCard(titleText, preferredHeight)
    local card = Instance.new("Frame")
    card.AutomaticSize = Enum.AutomaticSize.Y
    card.Size = UDim2.new(1, 0, 0, preferredHeight or 0)
    card.BorderSizePixel = 0
    card.Parent = AgentUI.ContextFrame

    Style.applyGlassEffect(card, {
        transparency = 0.25,
        baseColor = Color3.fromRGB(20, 24, 56),
        cornerRadius = 12,
        gradientRotation = 100,
        gradientTransparency = NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.08),
            NumberSequenceKeypoint.new(1, 0.4)
        })
    })

    local padding = Instance.new("UIPadding")
    padding.PaddingTop = UDim.new(0, 10)
    padding.PaddingBottom = UDim.new(0, 10)
    padding.PaddingLeft = UDim.new(0, 12)
    padding.PaddingRight = UDim.new(0, 12)
    padding.Parent = card

    local header = Instance.new("TextLabel")
    header.BackgroundTransparency = 1
    header.Size = UDim2.new(1, 0, 0, 20)
    header.Font = Enum.Font.GothamSemibold
    header.TextSize = 15
    header.TextColor3 = Color3.fromRGB(220, 230, 255)
    header.TextXAlignment = Enum.TextXAlignment.Left
    header.Text = titleText
    header.Parent = card
    card:SetAttribute("ContextHeader", true)

    return card, header
end

local SnapshotCard, SnapshotHeader = createContextCard("Snapshot", 110)
AgentUI.SnapshotCard = SnapshotCard
AgentUI.SnapshotHeader = SnapshotHeader

AgentUI.ContextButtonRow = Instance.new("Frame")
AgentUI.ContextButtonRow.Name = "ContextButtonRow"
AgentUI.ContextButtonRow.Size = UDim2.new(1, 0, 0, 26)
AgentUI.ContextButtonRow.BackgroundTransparency = 1
AgentUI.ContextButtonRow.Parent = SnapshotCard

AgentUI.ContextButtonLayout = Instance.new("UIListLayout")
AgentUI.ContextButtonLayout.FillDirection = Enum.FillDirection.Horizontal
AgentUI.ContextButtonLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.ContextButtonLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.ContextButtonLayout.Padding = UDim.new(0, 8)
AgentUI.ContextButtonLayout.Parent = AgentUI.ContextButtonRow

AgentUI.RefreshContextButton = Instance.new("TextButton")
AgentUI.RefreshContextButton.Name = "RefreshContextButton"
AgentUI.RefreshContextButton.Size = UDim2.new(0, 150, 0, 24)
AgentUI.RefreshContextButton.Font = Enum.Font.GothamSemibold
AgentUI.RefreshContextButton.TextSize = 13
AgentUI.RefreshContextButton.AutoButtonColor = false
AgentUI.RefreshContextButton.Text = "Refresh Snapshot"
AgentUI.RefreshContextButton.Parent = AgentUI.ContextButtonRow
Style.stylePrimaryButton(AgentUI.RefreshContextButton)

AgentUI.SnapshotTimestampLabel = Instance.new("TextLabel")
AgentUI.SnapshotTimestampLabel.Name = "SnapshotTimestampLabel"
AgentUI.SnapshotTimestampLabel.BackgroundTransparency = 1
AgentUI.SnapshotTimestampLabel.Size = UDim2.new(1, 0, 0, 18)
AgentUI.SnapshotTimestampLabel.Position = UDim2.new(0, 0, 0, 28)
AgentUI.SnapshotTimestampLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.SnapshotTimestampLabel.TextColor3 = Color3.fromRGB(73, 80, 87)
AgentUI.SnapshotTimestampLabel.Font = Enum.Font.SourceSans
AgentUI.SnapshotTimestampLabel.TextSize = 13
AgentUI.SnapshotTimestampLabel.Text = "Snapshot: not captured"
AgentUI.SnapshotTimestampLabel.Parent = SnapshotCard

AgentUI.SnapshotSummaryLabel = Instance.new("TextLabel")
AgentUI.SnapshotSummaryLabel.Name = "SnapshotSummary"
AgentUI.SnapshotSummaryLabel.BackgroundTransparency = 1
AgentUI.SnapshotSummaryLabel.Size = UDim2.new(1, 0, 0, 40)
AgentUI.SnapshotSummaryLabel.Position = UDim2.new(0, 0, 0, 50)
AgentUI.SnapshotSummaryLabel.TextWrapped = true
AgentUI.SnapshotSummaryLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.SnapshotSummaryLabel.TextYAlignment = Enum.TextYAlignment.Top
AgentUI.SnapshotSummaryLabel.Font = Enum.Font.SourceSans
AgentUI.SnapshotSummaryLabel.TextSize = 12
AgentUI.SnapshotSummaryLabel.TextColor3 = Color3.fromRGB(55, 65, 81)
AgentUI.SnapshotSummaryLabel.Text = "No recent hierarchy summary."
AgentUI.SnapshotSummaryLabel.Parent = SnapshotCard

-- Error Fix Button (shown when playtest stops with errors)
AgentUI.ErrorFixButton = Instance.new("TextButton")
AgentUI.ErrorFixButton.Name = "ErrorFixButton"
AgentUI.ErrorFixButton.Size = UDim2.new(1, 0, 0, 36)
AgentUI.ErrorFixButton.Position = UDim2.new(0, 0, 0, 95)
AgentUI.ErrorFixButton.Text = "Fix the error?"
AgentUI.ErrorFixButton.Font = Enum.Font.GothamSemibold
AgentUI.ErrorFixButton.TextSize = 13
AgentUI.ErrorFixButton.AutoButtonColor = false
AgentUI.ErrorFixButton.Visible = false
AgentUI.ErrorFixButton.TextColor3 = Color3.fromRGB(255, 255, 255)
AgentUI.ErrorFixButton.Parent = SnapshotCard
Style.applyGlassEffect(AgentUI.ErrorFixButton, {
    transparency = 0.15,
    baseColor = Color3.fromRGB(220, 53, 69),
    cornerRadius = 10,
    gradientRotation = 120
})
local errorFixPadding = Instance.new("UIPadding")
errorFixPadding.PaddingTop = UDim.new(0, 8)
errorFixPadding.PaddingBottom = UDim.new(0, 8)
errorFixPadding.PaddingLeft = UDim.new(0, 12)
errorFixPadding.PaddingRight = UDim.new(0, 12)
errorFixPadding.Parent = AgentUI.ErrorFixButton

-- Real-time Error Monitoring Card (shown during playtesting)
local ErrorMonitorCard = Instance.new("Frame")
ErrorMonitorCard.Name = "ErrorMonitorCard"
ErrorMonitorCard.Size = UDim2.new(1, 0, 0, 70)
ErrorMonitorCard.BackgroundTransparency = 1
ErrorMonitorCard.Visible = false
ErrorMonitorCard.LayoutOrder = 0
ErrorMonitorCard.Parent = AgentUI.Workspace

Style.applyGlassEffect(ErrorMonitorCard, {
    transparency = 0.12,
    baseColor = Color3.fromRGB(220, 53, 69),
    cornerRadius = 12,
    gradientRotation = 90
})

local errorMonitorPadding = Instance.new("UIPadding")
errorMonitorPadding.PaddingTop = UDim.new(0, 10)
errorMonitorPadding.PaddingBottom = UDim.new(0, 10)
errorMonitorPadding.PaddingLeft = UDim.new(0, 12)
errorMonitorPadding.PaddingRight = UDim.new(0, 12)
errorMonitorPadding.Parent = ErrorMonitorCard

local errorMonitorLayout = Instance.new("UIListLayout")
errorMonitorLayout.FillDirection = Enum.FillDirection.Vertical
errorMonitorLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
errorMonitorLayout.SortOrder = Enum.SortOrder.LayoutOrder
errorMonitorLayout.Padding = UDim.new(0, 6)
errorMonitorLayout.Parent = ErrorMonitorCard

AgentUI.ErrorCountLabel = Instance.new("TextLabel")
AgentUI.ErrorCountLabel.Name = "ErrorCountLabel"
AgentUI.ErrorCountLabel.Size = UDim2.new(1, 0, 0, 20)
AgentUI.ErrorCountLabel.BackgroundTransparency = 1
AgentUI.ErrorCountLabel.Font = Enum.Font.GothamSemibold
AgentUI.ErrorCountLabel.TextSize = 14
AgentUI.ErrorCountLabel.TextColor3 = Color3.fromRGB(255, 255, 255)
AgentUI.ErrorCountLabel.TextXAlignment = Enum.TextXAlignment.Left
AgentUI.ErrorCountLabel.Text = "0 errors found"
AgentUI.ErrorCountLabel.Parent = ErrorMonitorCard

AgentUI.ErrorDebugButton = Instance.new("TextButton")
AgentUI.ErrorDebugButton.Name = "ErrorDebugButton"
AgentUI.ErrorDebugButton.Size = UDim2.new(1, 0, 0, 28)
AgentUI.ErrorDebugButton.Font = Enum.Font.GothamSemibold
AgentUI.ErrorDebugButton.TextSize = 12
AgentUI.ErrorDebugButton.AutoButtonColor = false
AgentUI.ErrorDebugButton.Text = "Debug with Agent?"
AgentUI.ErrorDebugButton.TextColor3 = Color3.fromRGB(255, 255, 255)
AgentUI.ErrorDebugButton.Parent = ErrorMonitorCard
Style.applyGlassEffect(AgentUI.ErrorDebugButton, {
    transparency = 0.2,
    baseColor = Color3.fromRGB(40, 167, 69),
    cornerRadius = 8,
    gradientRotation = 120
})

local errorDebugPadding = Instance.new("UIPadding")
errorDebugPadding.PaddingTop = UDim.new(0, 6)
errorDebugPadding.PaddingBottom = UDim.new(0, 6)
errorDebugPadding.PaddingLeft = UDim.new(0, 10)
errorDebugPadding.PaddingRight = UDim.new(0, 10)
errorDebugPadding.Parent = AgentUI.ErrorDebugButton

local DiffCard, DiffHeader = createContextCard("Recent Changes", 160)
AgentUI.DiffCard = DiffCard
AgentUI.DiffHeader = DiffHeader

AgentUI.DiffScroll = Instance.new("ScrollingFrame")
AgentUI.DiffScroll.Name = "DiffScroll"
AgentUI.DiffScroll.Size = UDim2.new(1, 0, 1, -26)
AgentUI.DiffScroll.Position = UDim2.new(0, 0, 0, 26)
AgentUI.DiffScroll.BackgroundTransparency = 1
AgentUI.DiffScroll.BorderSizePixel = 0
AgentUI.DiffScroll.AutomaticCanvasSize = Enum.AutomaticSize.None
AgentUI.DiffScroll.CanvasSize = UDim2.new(0, 0, 0, 0)
AgentUI.DiffScroll.ScrollBarThickness = 5
AgentUI.DiffScroll.ScrollBarImageColor3 = Color3.fromRGB(102, 126, 234)
AgentUI.DiffScroll.Parent = DiffCard

AgentUI.DiffLayout = Instance.new("UIListLayout")
AgentUI.DiffLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.DiffLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.DiffLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.DiffLayout.Padding = UDim.new(0, 4)
AgentUI.DiffLayout.Parent = AgentUI.DiffScroll

local function updateDiffCanvasSize()
    AgentUI.DiffScroll.CanvasSize = UDim2.new(0, 0, 0, AgentUI.DiffLayout.AbsoluteContentSize.Y + 6)
end

AgentUI.DiffLayout:GetPropertyChangedSignal("AbsoluteContentSize"):Connect(updateDiffCanvasSize)
task.defer(updateDiffCanvasSize)

local PlanCard, PlanHeader = createContextCard("Active Plan", 190)
AgentUI.PlanCard = PlanCard
AgentUI.PlanHeader = PlanHeader

AgentUI.PlannerScroll = Instance.new("ScrollingFrame")
AgentUI.PlannerScroll.Name = "PlannerScroll"
AgentUI.PlannerScroll.Size = UDim2.new(1, 0, 1, -26)
AgentUI.PlannerScroll.Position = UDim2.new(0, 0, 0, 26)
AgentUI.PlannerScroll.BackgroundTransparency = 1
AgentUI.PlannerScroll.BorderSizePixel = 0
AgentUI.PlannerScroll.AutomaticCanvasSize = Enum.AutomaticSize.None
AgentUI.PlannerScroll.CanvasSize = UDim2.new(0, 0, 0, 0)
AgentUI.PlannerScroll.ScrollBarThickness = 5
AgentUI.PlannerScroll.ScrollBarImageColor3 = Color3.fromRGB(102, 126, 234)
AgentUI.PlannerScroll.Parent = PlanCard

AgentUI.PlannerLayout = Instance.new("UIListLayout")
AgentUI.PlannerLayout.FillDirection = Enum.FillDirection.Vertical
AgentUI.PlannerLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
AgentUI.PlannerLayout.SortOrder = Enum.SortOrder.LayoutOrder
AgentUI.PlannerLayout.Padding = UDim.new(0, 6)
AgentUI.PlannerLayout.Parent = AgentUI.PlannerScroll

local function updatePlannerCanvasSize()
    AgentUI.PlannerScroll.CanvasSize = UDim2.new(0, 0, 0, AgentUI.PlannerLayout.AbsoluteContentSize.Y + 6)
end

AgentUI.PlannerLayout:GetPropertyChangedSignal("AbsoluteContentSize"):Connect(updatePlannerCanvasSize)
task.defer(updatePlannerCanvasSize)

local activeTab = "Agent"
local tabActiveGradient = ColorSequence.new({
    ColorSequenceKeypoint.new(0, Color3.fromRGB(178, 148, 255)),
    ColorSequenceKeypoint.new(1, Color3.fromRGB(96, 128, 255))
})
local tabInactiveGradient = ColorSequence.new({
    ColorSequenceKeypoint.new(0, Color3.fromRGB(40, 48, 110)),
    ColorSequenceKeypoint.new(1, Color3.fromRGB(32, 36, 88))
})

local function setTabStyle(button, active)
    local gradient = button:FindFirstChild("TabGradient")
    if gradient then
        gradient.Color = active and tabActiveGradient or tabInactiveGradient
        gradient.Transparency = active and NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.08),
            NumberSequenceKeypoint.new(1, 0.25)
        }) or NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.3),
            NumberSequenceKeypoint.new(1, 0.55)
        })
    end

    local stroke = button:FindFirstChild("TabStroke")
    if stroke then
        stroke.Transparency = active and 0.15 or 0.45
    end

    button.TextColor3 = Color3.fromRGB(255, 255, 255)
end

local function updateTabVisuals()
    local settingsActive = activeTab == "Settings"
    local agentActive = activeTab == "Agent"
    local vfxActive = activeTab == "VFX"
    local hasApiKey = false
    if type(getActiveApiKey) == "function" then
        hasApiKey = getActiveApiKey() ~= ""
    end

    setTabStyle(SettingsTabButton, settingsActive)
    setTabStyle(AgentTabButton, agentActive)
    if VFXTabButton then
        setTabStyle(VFXTabButton, vfxActive)
        VFXTabButton.Visible = hasApiKey
    end

    SettingsPage.Visible = settingsActive
    AgentPage.Visible = agentActive
    if VFXPage then
        VFXPage.Visible = vfxActive and hasApiKey
    end
end

-- Update VFX tab visibility when API key changes
local function updateVFXTabVisibility()
    if not VFXTabButton or type(getActiveApiKey) ~= "function" then return end
    local hasApiKey = getActiveApiKey() ~= ""
    VFXTabButton.Visible = hasApiKey
    if not hasApiKey and activeTab == "VFX" then
        activeTab = "Agent"
        updateTabVisuals()
    end
end

SettingsTabButton.MouseButton1Click:Connect(function()
    activeTab = "Settings"
    updateTabVisuals()
end)

AgentTabButton.MouseButton1Click:Connect(function()
    activeTab = "Agent"
    updateTabVisuals()
end)

VFXTabButton.MouseButton1Click:Connect(function()
    if type(getActiveApiKey) == "function" then
        local apiKey = getActiveApiKey()
        if apiKey == "" then
            showApiKeyPopup()
            return
        end
        activeTab = "VFX"
        updateTabVisuals()
        task.wait(0.2)
        VFXUI.load()
    end
end)

updateTabVisuals()

-- Legacy aliases for agent UI compatibility
ChatScroll = AgentUI.ChatScroll
ChatInputBox = AgentUI.ChatInputBox
ChatInputFrame = AgentUI.ChatInputFrame
SendButton = AgentUI.SendButton
ContextFrame = AgentUI.ContextFrame
RefreshContextButton = AgentUI.RefreshContextButton
SnapshotTimestampLabel = AgentUI.SnapshotTimestampLabel
SnapshotSummary = AgentUI.SnapshotSummaryLabel
DiffScroll = AgentUI.DiffScroll
PlannerScroll = AgentUI.PlannerScroll
ModelSelectorFrame = AgentUI.ModelSelectorFrame
TrustToggle = AgentUI.TrustToggle

-- Create title
local HeaderFrame = Instance.new("Frame")
HeaderFrame.Name = "HeaderFrame"
HeaderFrame.Size = UDim2.new(1, -20, 0, TitleHeight)
HeaderFrame.Position = UDim2.new(0, 10, 0, 6)
HeaderFrame.BorderSizePixel = 0
HeaderFrame.Parent = TopFrame
Style.applyGlassEffect(HeaderFrame, {
    transparency = 0.15,
    baseColor = Color3.fromRGB(20, 24, 60),
    cornerRadius = 14,
    gradientRotation = 120,
    gradientTransparency = NumberSequence.new({
        NumberSequenceKeypoint.new(0, 0.05),
        NumberSequenceKeypoint.new(1, 0.35)
    })
})

local Title = Instance.new("TextLabel")
Title.Size = UDim2.new(1, -24, 1, 0)
Title.Position = UDim2.new(0, 12, 0, 0)
Title.BackgroundTransparency = 1
Title.TextColor3 = Color3.fromRGB(235, 240, 255)
Title.Text = "Electrode"
Title.TextScaled = false
Title.TextSize = 26
Title.TextXAlignment = Enum.TextXAlignment.Left
Title.TextYAlignment = Enum.TextYAlignment.Center
Title.Font = Enum.Font.GothamBold
Title.Parent = HeaderFrame

local TitleGlow = Style.ensureChild(HeaderFrame, "UIStroke", "TitleStroke")
TitleGlow.Thickness = 1
TitleGlow.Color = Color3.fromRGB(140, 150, 255)
TitleGlow.Transparency = 0.4

-- Settings Page Layout
SettingsUI.Padding = Instance.new("UIPadding")
SettingsUI.Padding.PaddingTop = UDim.new(0, 20)
SettingsUI.Padding.PaddingBottom = UDim.new(0, 20)
SettingsUI.Padding.PaddingLeft = UDim.new(0, 20)
SettingsUI.Padding.PaddingRight = UDim.new(0, 20)
SettingsUI.Padding.Parent = SettingsPage

SettingsUI.Layout = Instance.new("UIListLayout")
SettingsUI.Layout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.Layout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.Layout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.Layout.Padding = UDim.new(0, 16)
SettingsUI.Layout.Parent = SettingsPage

-- Settings Title
SettingsUI.Title = Instance.new("TextLabel")
SettingsUI.Title.Size = UDim2.new(1, 0, 0, 30)
SettingsUI.Title.BackgroundTransparency = 1
SettingsUI.Title.Text = "Settings"
SettingsUI.Title.TextColor3 = Color3.fromRGB(235, 240, 255)
SettingsUI.Title.TextScaled = false
SettingsUI.Title.TextSize = 24
SettingsUI.Title.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.Title.Font = Enum.Font.GothamBold
SettingsUI.Title.LayoutOrder = 1
SettingsUI.Title.Parent = SettingsPage

-- API Key Section removed - API keys are entered via popup only

-- API Key Popup/Modal (shows when API key is missing)
local apiKeyPopup = Instance.new("Frame")
apiKeyPopup.Name = "ApiKeyPopup"
apiKeyPopup.Size = UDim2.new(0.85, 0, 0, 0)
apiKeyPopup.AutomaticSize = Enum.AutomaticSize.Y
apiKeyPopup.Position = UDim2.new(0.5, 0, 0.5, 0)
apiKeyPopup.AnchorPoint = Vector2.new(0.5, 0.5)
apiKeyPopup.ZIndex = 100
apiKeyPopup.Visible = false
apiKeyPopup.Parent = MainFrame
Style.applyGlassEffect(apiKeyPopup, {
    transparency = 0.1,
    baseColor = Color3.fromRGB(18, 22, 58),
    cornerRadius = 16,
    gradientRotation = 120
})

local apiKeyPopupPadding = Instance.new("UIPadding")
apiKeyPopupPadding.PaddingTop = UDim.new(0, 24)
apiKeyPopupPadding.PaddingBottom = UDim.new(0, 24)
apiKeyPopupPadding.PaddingLeft = UDim.new(0, 24)
apiKeyPopupPadding.PaddingRight = UDim.new(0, 24)
apiKeyPopupPadding.Parent = apiKeyPopup

local apiKeyPopupLayout = Instance.new("UIListLayout")
apiKeyPopupLayout.FillDirection = Enum.FillDirection.Vertical
apiKeyPopupLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
apiKeyPopupLayout.SortOrder = Enum.SortOrder.LayoutOrder
apiKeyPopupLayout.Padding = UDim.new(0, 16)
apiKeyPopupLayout.Parent = apiKeyPopup

local apiKeyPopupTitle = Instance.new("TextLabel")
apiKeyPopupTitle.Size = UDim2.new(1, 0, 0, 28)
apiKeyPopupTitle.BackgroundTransparency = 1
apiKeyPopupTitle.Text = "API Key Required"
apiKeyPopupTitle.TextColor3 = Color3.fromRGB(235, 240, 255)
apiKeyPopupTitle.TextScaled = false
apiKeyPopupTitle.TextSize = 22
apiKeyPopupTitle.TextXAlignment = Enum.TextXAlignment.Left
apiKeyPopupTitle.Font = Enum.Font.GothamBold
apiKeyPopupTitle.LayoutOrder = 1
apiKeyPopupTitle.Parent = apiKeyPopup

local apiKeyPopupMessage = Instance.new("TextLabel")
apiKeyPopupMessage.Size = UDim2.new(1, 0, 0, 0)
apiKeyPopupMessage.AutomaticSize = Enum.AutomaticSize.Y
apiKeyPopupMessage.BackgroundTransparency = 1
apiKeyPopupMessage.TextWrapped = true
apiKeyPopupMessage.Text = "Electrode Agent requires an API key to function. Please enter your API key below."
apiKeyPopupMessage.TextColor3 = Color3.fromRGB(200, 208, 245)
apiKeyPopupMessage.TextScaled = false
apiKeyPopupMessage.TextSize = 14
apiKeyPopupMessage.TextXAlignment = Enum.TextXAlignment.Left
apiKeyPopupMessage.Font = Enum.Font.Gotham
apiKeyPopupMessage.LayoutOrder = 2
apiKeyPopupMessage.Parent = apiKeyPopup

-- API Key Input Container
local apiKeyPopupInputContainer = Instance.new("Frame")
apiKeyPopupInputContainer.Size = UDim2.new(1, 0, 0, 30)
apiKeyPopupInputContainer.BackgroundTransparency = 1
apiKeyPopupInputContainer.LayoutOrder = 3
apiKeyPopupInputContainer.Parent = apiKeyPopup

local apiKeyPopupInput = Instance.new("TextBox")
apiKeyPopupInput.Size = UDim2.new(1, 0, 1, 0)
apiKeyPopupInput.Text = ""
apiKeyPopupInput.PlaceholderText = "Enter your API key..."
apiKeyPopupInput.TextScaled = false
apiKeyPopupInput.TextSize = 14
apiKeyPopupInput.Font = Enum.Font.Gotham
apiKeyPopupInput.Parent = apiKeyPopupInputContainer
Style.styleInputBox(apiKeyPopupInput)

-- API Key Status Label
local apiKeyPopupStatusLabel = Instance.new("TextLabel")
apiKeyPopupStatusLabel.Size = UDim2.new(1, 0, 0, 20)
apiKeyPopupStatusLabel.BackgroundTransparency = 1
apiKeyPopupStatusLabel.Text = ""
apiKeyPopupStatusLabel.TextColor3 = Color3.fromRGB(200, 208, 245)
apiKeyPopupStatusLabel.TextScaled = false
apiKeyPopupStatusLabel.TextSize = 13
apiKeyPopupStatusLabel.TextXAlignment = Enum.TextXAlignment.Left
apiKeyPopupStatusLabel.Font = Enum.Font.Gotham
apiKeyPopupStatusLabel.LayoutOrder = 4
apiKeyPopupStatusLabel.Parent = apiKeyPopup

local apiKeyPopupButton = Instance.new("TextButton")
apiKeyPopupButton.Size = UDim2.new(0, 200, 0, 36)
apiKeyPopupButton.Text = "Submit"
apiKeyPopupButton.TextColor3 = Color3.fromRGB(255, 255, 255)
apiKeyPopupButton.TextScaled = false
apiKeyPopupButton.TextSize = 15
apiKeyPopupButton.Font = Enum.Font.GothamSemibold
apiKeyPopupButton.LayoutOrder = 5
apiKeyPopupButton.Parent = apiKeyPopup
Style.stylePrimaryButton(apiKeyPopupButton)

apiKeyPopupButton.MouseButton1Click:Connect(function()
    local apiKey = apiKeyPopupInput.Text
    
    if apiKey == "" then
        apiKeyPopupStatusLabel.Text = "Please enter your API key"
        apiKeyPopupStatusLabel.TextColor3 = STATUS_COLORS.error
        return
    end
    
    -- Update UI
    apiKeyPopupButton.Text = "Validating..."
    apiKeyPopupStatusLabel.Text = "Validating API key..."
    apiKeyPopupStatusLabel.TextColor3 = STATUS_COLORS.info
    
    -- Validate API key
    if Network.validateApiKey(apiKey) then
        -- Save API key
        pcall(function()
            plugin:SetSetting("ApiKey", apiKey)
        end)
        cachedApiKey = apiKey
        updateVFXTabVisibility()
        
        apiKeyPopupStatusLabel.Text = "API key validated successfully!"
        apiKeyPopupStatusLabel.TextColor3 = STATUS_COLORS.success
        
        task.wait(1.5)
        apiKeyPopup.Visible = false
        apiKeyPopupStatusLabel.Text = ""
        apiKeyPopupInput.Text = ""
    else
        apiKeyPopupStatusLabel.Text = "Invalid API key. Please check your key and try again."
        apiKeyPopupStatusLabel.TextColor3 = STATUS_COLORS.error
        apiKeyPopupButton.Text = "Submit"
    end
end)

-- Enter key support for API key input
apiKeyPopupInput.FocusLost:Connect(function(enterPressed)
    if enterPressed then
        apiKeyPopupButton.MouseButton1Click:Fire()
    end
end)

function showApiKeyPopup()
    if apiKeyPopup then
        apiKeyPopup.Visible = true
        -- Clear previous input and focus
        if apiKeyPopupInput then
            apiKeyPopupInput.Text = ""
            task.wait(0.1)
            apiKeyPopupInput:CaptureFocus()
        end
        -- Clear status label
        if apiKeyPopupStatusLabel then
            apiKeyPopupStatusLabel.Text = ""
        end
    end
end

function hideApiKeyPopup()
    if apiKeyPopup then
        apiKeyPopup.Visible = false
    end
end

-- API key locking/change functionality removed - API keys are only entered via popup

-- Additional Settings Sections

-- Auto-save Chat History Toggle
SettingsUI.AutoSaveSection = Instance.new("Frame")
SettingsUI.AutoSaveSection.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.AutoSaveSection.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.AutoSaveSection.BackgroundTransparency = 1
SettingsUI.AutoSaveSection.LayoutOrder = 2
SettingsUI.AutoSaveSection.Parent = SettingsPage

SettingsUI.AutoSaveSectionLayout = Instance.new("UIListLayout")
SettingsUI.AutoSaveSectionLayout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.AutoSaveSectionLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.AutoSaveSectionLayout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.AutoSaveSectionLayout.Padding = UDim.new(0, 12)
SettingsUI.AutoSaveSectionLayout.Parent = SettingsUI.AutoSaveSection

SettingsUI.AutoSaveLabel = Instance.new("TextLabel")
SettingsUI.AutoSaveLabel.Size = UDim2.new(1, 0, 0, 20)
SettingsUI.AutoSaveLabel.BackgroundTransparency = 1
SettingsUI.AutoSaveLabel.Text = "Auto-save Chat History"
SettingsUI.AutoSaveLabel.TextColor3 = Color3.fromRGB(220, 230, 255)
SettingsUI.AutoSaveLabel.TextScaled = false
SettingsUI.AutoSaveLabel.TextSize = 14
SettingsUI.AutoSaveLabel.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.AutoSaveLabel.Font = Enum.Font.GothamSemibold
SettingsUI.AutoSaveLabel.LayoutOrder = 1
SettingsUI.AutoSaveLabel.Parent = SettingsUI.AutoSaveSection

SettingsUI.AutoSaveToggle = Instance.new("TextButton")
SettingsUI.AutoSaveToggle.Size = UDim2.new(0, 50, 0, 26)
SettingsUI.AutoSaveToggle.Text = ""
SettingsUI.AutoSaveToggle.AutoButtonColor = false
SettingsUI.AutoSaveToggle.LayoutOrder = 2
SettingsUI.AutoSaveToggle.Parent = SettingsUI.AutoSaveSection
Style.applyGlassEffect(SettingsUI.AutoSaveToggle, {
    transparency = 0.3,
    baseColor = Color3.fromRGB(22, 26, 62),
    cornerRadius = 13,
    gradientRotation = 120
})

local autoSaveToggleIndicator = Instance.new("Frame")
autoSaveToggleIndicator.Name = "Indicator"
autoSaveToggleIndicator.Size = UDim2.new(0, 20, 0, 20)
autoSaveToggleIndicator.Position = UDim2.new(0, 3, 0.5, 0)
autoSaveToggleIndicator.AnchorPoint = Vector2.new(0, 0.5)
autoSaveToggleIndicator.BackgroundColor3 = Color3.fromRGB(200, 200, 200)
autoSaveToggleIndicator.BorderSizePixel = 0
autoSaveToggleIndicator.Parent = SettingsUI.AutoSaveToggle

local autoSaveToggleCorner = Instance.new("UICorner")
autoSaveToggleCorner.CornerRadius = UDim.new(0, 10)
autoSaveToggleCorner.Parent = autoSaveToggleIndicator

SettingsUI.AutoSaveToggle:SetAttribute("Enabled", true)

local function updateAutoSaveToggle()
    local enabled = SettingsUI.AutoSaveToggle:GetAttribute("Enabled") == true
    if enabled then
        autoSaveToggleIndicator.Position = UDim2.new(1, -23, 0.5, 0)
        autoSaveToggleIndicator.BackgroundColor3 = Color3.fromRGB(96, 128, 255)
    else
        autoSaveToggleIndicator.Position = UDim2.new(0, 3, 0.5, 0)
        autoSaveToggleIndicator.BackgroundColor3 = Color3.fromRGB(200, 200, 200)
    end
end

SettingsUI.AutoSaveToggle.MouseButton1Click:Connect(function()
    local current = SettingsUI.AutoSaveToggle:GetAttribute("Enabled") == true
    SettingsUI.AutoSaveToggle:SetAttribute("Enabled", not current)
    updateAutoSaveToggle()
    
    -- Save preference
    pcall(function()
        plugin:SetSetting("AutoSaveChat", not current)
    end)
end)

-- Load auto-save preference
local function loadAutoSavePreference()
    local success, saved = pcall(function()
        return plugin:GetSetting("AutoSaveChat")
    end)
    if success and saved ~= nil then
        SettingsUI.AutoSaveToggle:SetAttribute("Enabled", saved)
    else
        SettingsUI.AutoSaveToggle:SetAttribute("Enabled", true) -- Default to enabled
    end
    updateAutoSaveToggle()
end

loadAutoSavePreference()

-- Clear Chat History Button
SettingsUI.ClearChatSection = Instance.new("Frame")
SettingsUI.ClearChatSection.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.ClearChatSection.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.ClearChatSection.BackgroundTransparency = 1
SettingsUI.ClearChatSection.LayoutOrder = 4
SettingsUI.ClearChatSection.Parent = SettingsPage

SettingsUI.ClearChatSectionLayout = Instance.new("UIListLayout")
SettingsUI.ClearChatSectionLayout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.ClearChatSectionLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.ClearChatSectionLayout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.ClearChatSectionLayout.Padding = UDim.new(0, 12)
SettingsUI.ClearChatSectionLayout.Parent = SettingsUI.ClearChatSection

SettingsUI.ClearChatLabel = Instance.new("TextLabel")
SettingsUI.ClearChatLabel.Size = UDim2.new(1, 0, 0, 20)
SettingsUI.ClearChatLabel.BackgroundTransparency = 1
SettingsUI.ClearChatLabel.Text = "Chat History"
SettingsUI.ClearChatLabel.TextColor3 = Color3.fromRGB(220, 230, 255)
SettingsUI.ClearChatLabel.TextScaled = false
SettingsUI.ClearChatLabel.TextSize = 14
SettingsUI.ClearChatLabel.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.ClearChatLabel.Font = Enum.Font.GothamSemibold
SettingsUI.ClearChatLabel.LayoutOrder = 1
SettingsUI.ClearChatLabel.Parent = SettingsUI.ClearChatSection

SettingsUI.ClearChatButton = Instance.new("TextButton")
SettingsUI.ClearChatButton.Size = UDim2.new(0, 150, 0, 32)
SettingsUI.ClearChatButton.Text = "Clear Chat History"
SettingsUI.ClearChatButton.TextColor3 = Color3.fromRGB(255, 255, 255)
SettingsUI.ClearChatButton.TextScaled = false
SettingsUI.ClearChatButton.TextSize = 13
SettingsUI.ClearChatButton.Font = Enum.Font.GothamSemibold
SettingsUI.ClearChatButton.LayoutOrder = 2
SettingsUI.ClearChatButton.Parent = SettingsUI.ClearChatSection
Style.styleSecondaryButton(SettingsUI.ClearChatButton)

SettingsUI.ClearChatButton.MouseButton1Click:Connect(function()
    if clearAgentChat then
        clearAgentChat()
        -- Status message removed since there's no status label in Settings anymore
    end
end)

-- Show Context Cards Toggle
SettingsUI.ContextCardsSection = Instance.new("Frame")
SettingsUI.ContextCardsSection.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.ContextCardsSection.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.ContextCardsSection.BackgroundTransparency = 1
SettingsUI.ContextCardsSection.LayoutOrder = 5
SettingsUI.ContextCardsSection.Parent = SettingsPage

SettingsUI.ContextCardsSectionLayout = Instance.new("UIListLayout")
SettingsUI.ContextCardsSectionLayout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.ContextCardsSectionLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.ContextCardsSectionLayout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.ContextCardsSectionLayout.Padding = UDim.new(0, 12)
SettingsUI.ContextCardsSectionLayout.Parent = SettingsUI.ContextCardsSection

SettingsUI.ContextCardsLabel = Instance.new("TextLabel")
SettingsUI.ContextCardsLabel.Size = UDim2.new(1, 0, 0, 20)
SettingsUI.ContextCardsLabel.BackgroundTransparency = 1
SettingsUI.ContextCardsLabel.Text = "Show Context Cards"
SettingsUI.ContextCardsLabel.TextColor3 = Color3.fromRGB(220, 230, 255)
SettingsUI.ContextCardsLabel.TextScaled = false
SettingsUI.ContextCardsLabel.TextSize = 14
SettingsUI.ContextCardsLabel.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.ContextCardsLabel.Font = Enum.Font.GothamSemibold
SettingsUI.ContextCardsLabel.LayoutOrder = 1
SettingsUI.ContextCardsLabel.Parent = SettingsUI.ContextCardsSection

SettingsUI.ContextCardsToggle = Instance.new("TextButton")
SettingsUI.ContextCardsToggle.Size = UDim2.new(0, 50, 0, 26)
SettingsUI.ContextCardsToggle.Text = ""
SettingsUI.ContextCardsToggle.AutoButtonColor = false
SettingsUI.ContextCardsToggle.LayoutOrder = 2
SettingsUI.ContextCardsToggle.Parent = SettingsUI.ContextCardsSection
Style.applyGlassEffect(SettingsUI.ContextCardsToggle, {
    transparency = 0.3,
    baseColor = Color3.fromRGB(22, 26, 62),
    cornerRadius = 13,
    gradientRotation = 120
})

local contextCardsToggleIndicator = Instance.new("Frame")
contextCardsToggleIndicator.Name = "Indicator"
contextCardsToggleIndicator.Size = UDim2.new(0, 20, 0, 20)
contextCardsToggleIndicator.Position = UDim2.new(0, 3, 0.5, 0)
contextCardsToggleIndicator.AnchorPoint = Vector2.new(0, 0.5)
contextCardsToggleIndicator.BackgroundColor3 = Color3.fromRGB(200, 200, 200)
contextCardsToggleIndicator.BorderSizePixel = 0
contextCardsToggleIndicator.Parent = SettingsUI.ContextCardsToggle

local contextCardsToggleCorner = Instance.new("UICorner")
contextCardsToggleCorner.CornerRadius = UDim.new(0, 10)
contextCardsToggleCorner.Parent = contextCardsToggleIndicator

SettingsUI.ContextCardsToggle:SetAttribute("Enabled", true)

local function updateContextCardsToggle()
    local enabled = SettingsUI.ContextCardsToggle:GetAttribute("Enabled") == true
    if enabled then
        contextCardsToggleIndicator.Position = UDim2.new(1, -23, 0.5, 0)
        contextCardsToggleIndicator.BackgroundColor3 = Color3.fromRGB(96, 128, 255)
        if AgentUI.ContextFrame then
            AgentUI.ContextFrame.Visible = true
        end
    else
        contextCardsToggleIndicator.Position = UDim2.new(0, 3, 0.5, 0)
        contextCardsToggleIndicator.BackgroundColor3 = Color3.fromRGB(200, 200, 200)
        if AgentUI.ContextFrame then
            AgentUI.ContextFrame.Visible = false
        end
    end
end

SettingsUI.ContextCardsToggle.MouseButton1Click:Connect(function()
    local current = SettingsUI.ContextCardsToggle:GetAttribute("Enabled") == true
    SettingsUI.ContextCardsToggle:SetAttribute("Enabled", not current)
    updateContextCardsToggle()
    
    -- Save preference
    pcall(function()
        plugin:SetSetting("ShowContextCards", not current)
    end)
end)

-- Load context cards preference
local function loadContextCardsPreference()
    local success, saved = pcall(function()
        return plugin:GetSetting("ShowContextCards")
    end)
    if success and saved ~= nil then
        SettingsUI.ContextCardsToggle:SetAttribute("Enabled", saved)
    else
        SettingsUI.ContextCardsToggle:SetAttribute("Enabled", true) -- Default to enabled
    end
    updateContextCardsToggle()
end

loadContextCardsPreference()

-- VFX Library UI Setup
VFXUI.Padding = Instance.new("UIPadding")
VFXUI.Padding.PaddingTop = UDim.new(0, 20)
VFXUI.Padding.PaddingBottom = UDim.new(0, 20)
VFXUI.Padding.PaddingLeft = UDim.new(0, 20)
VFXUI.Padding.PaddingRight = UDim.new(0, 20)
VFXUI.Padding.Parent = VFXPage

VFXUI.Layout = Instance.new("UIListLayout")
VFXUI.Layout.FillDirection = Enum.FillDirection.Vertical
VFXUI.Layout.Padding = UDim.new(0, 16)
VFXUI.Layout.SortOrder = Enum.SortOrder.LayoutOrder
VFXUI.Layout.Parent = VFXPage

VFXUI.Title = Instance.new("TextLabel")
VFXUI.Title.Size = UDim2.new(1, 0, 0, 30)
VFXUI.Title.BackgroundTransparency = 1
VFXUI.Title.Text = "Library"
VFXUI.Title.TextColor3 = Color3.fromRGB(235, 240, 255)
VFXUI.Title.TextSize = 24
VFXUI.Title.TextXAlignment = Enum.TextXAlignment.Left
VFXUI.Title.Font = Enum.Font.GothamBold
VFXUI.Title.LayoutOrder = 1
VFXUI.Title.Parent = VFXPage

VFXUI.ReminderLabel = Instance.new("TextLabel")
VFXUI.ReminderLabel.Size = UDim2.new(1, 0, 0, 50)
VFXUI.ReminderLabel.BackgroundTransparency = 1
VFXUI.ReminderLabel.Text = "Click a VFX card to reveal its Asset ID. Copy the ID and include it in your prompt to tell the AI to use it."
VFXUI.ReminderLabel.TextColor3 = Color3.fromRGB(180, 190, 220)
VFXUI.ReminderLabel.TextSize = 12
VFXUI.ReminderLabel.TextWrapped = true
VFXUI.ReminderLabel.TextXAlignment = Enum.TextXAlignment.Left
VFXUI.ReminderLabel.LayoutOrder = 2
VFXUI.ReminderLabel.Parent = VFXPage

VFXUI.ScrollFrame = Instance.new("ScrollingFrame")
VFXUI.ScrollFrame.Size = UDim2.new(1, 0, 1, -112)
VFXUI.ScrollFrame.Position = UDim2.new(0, 0, 0, 112)
VFXUI.ScrollFrame.BackgroundTransparency = 1
VFXUI.ScrollFrame.BorderSizePixel = 0
VFXUI.ScrollFrame.ScrollBarThickness = 8
VFXUI.ScrollFrame.ScrollBarImageColor3 = Color3.fromRGB(102, 126, 234)
VFXUI.ScrollFrame.AutomaticCanvasSize = Enum.AutomaticSize.Y
VFXUI.ScrollFrame.CanvasSize = UDim2.new(0, 0, 0, 0)
VFXUI.ScrollFrame.ZIndex = 1
VFXUI.ScrollFrame.Visible = true
VFXUI.ScrollFrame.Parent = VFXPage

local scrollPadding = Instance.new("UIPadding")
scrollPadding.PaddingTop = UDim.new(0, 0)
scrollPadding.PaddingBottom = UDim.new(0, 0)
scrollPadding.PaddingLeft = UDim.new(0, 0)
scrollPadding.PaddingRight = UDim.new(0, 0)
scrollPadding.Parent = VFXUI.ScrollFrame

VFXUI.Layout2 = Instance.new("UIListLayout")
VFXUI.Layout2.Padding = UDim.new(0, 10)
VFXUI.Layout2.FillDirection = Enum.FillDirection.Vertical
VFXUI.Layout2.Parent = VFXUI.ScrollFrame

VFXUI.assets = {
    {asset_id = "111715985018608", name = "PunchSmall", category = "Combat", description = "A compact punch impact effect perfect for melee combat. Creates a satisfying visual feedback when characters land punches."},
    {asset_id = "73398740744562", name = "PurpleExplosionBIG", category = "Explosions", description = "A large-scale purple explosion effect ideal for powerful abilities, magic spells, or dramatic moments. Creates an impressive visual impact."},
    {asset_id = "121639416384765", name = "ExplosionMedium", category = "Explosions", description = "A medium-sized explosion effect suitable for grenades, abilities, or environmental destruction. Balanced visual impact for most combat scenarios."},
    {asset_id = "71909210330723", name = "TornadoMedium", category = "Weather", description = "A swirling tornado effect perfect for weather-based abilities, wind magic, or environmental hazards. Creates dynamic movement and visual interest."},
    {asset_id = "74356088805119", name = "OverchargeHandEffectGreen", category = "Magic", description = "A green energy overcharge effect that emanates from hands. Perfect for charging abilities, power-ups, or magical transformations."},
    {asset_id = "124988278246548", name = "ElectricSlam", category = "Combat", description = "An electric slam impact effect combining lightning and impact. Great for electric-based melee attacks or ground slams with electrical properties."},
    {asset_id = "103270300816855", name = "FrozenDummyEffect", category = "Status", description = "A freezing effect that encases targets in ice. Perfect for status effects, debuffs, or ice-based abilities that immobilize enemies."},
    {asset_id = "139404871735759", name = "WhiteTornadoMedium", category = "Weather", description = "A white tornado effect with a cleaner, more ethereal appearance. Ideal for light magic, purification effects, or divine abilities."},
    {asset_id = "119127587640579", name = "WhiteElectricityMedium", category = "Magic", description = "White electrical energy effect perfect for lightning magic, divine powers, or high-energy abilities. Creates a bright, powerful visual."},
    {asset_id = "129272358397643", name = "GreenMultisplashBIG", category = "Magic", description = "A large green multisplash effect with multiple impact points. Great for area-of-effect abilities, poison attacks, or nature-based magic."},
    {asset_id = "93937430639513", name = "DummyHealth", category = "Status", description = "A health-related visual effect for healing, regeneration, or health indicators. Perfect for showing health restoration or status changes."}
}
VFXUI.selectedId = nil

function VFXUI.createCard(asset)
    if not VFXUI.ScrollFrame then return end
    
    local card = Instance.new("TextButton")
    card.Size = UDim2.new(1, 0, 0, 100)
    card.BackgroundColor3 = Color3.fromRGB(25, 30, 65)
    card.BorderSizePixel = 0
    card.Text = ""
    card.AutoButtonColor = false
    card.ZIndex = 1
    card.Parent = VFXUI.ScrollFrame
    
    local corner = Instance.new("UICorner")
    corner.CornerRadius = UDim.new(0, 12)
    corner.Parent = card
    
    local stroke = Instance.new("UIStroke")
    stroke.Thickness = 1.5
    stroke.Color = Color3.fromRGB(102, 126, 234)
    stroke.Transparency = 0.3
    stroke.Parent = card
    
    local gradient = Instance.new("UIGradient")
    gradient.Color = ColorSequence.new({
        ColorSequenceKeypoint.new(0, Color3.fromRGB(30, 35, 75)),
        ColorSequenceKeypoint.new(1, Color3.fromRGB(20, 25, 55))
    })
    gradient.Rotation = 90
    gradient.Parent = card
    
    local nameLabel = Instance.new("TextLabel")
    nameLabel.Size = UDim2.new(1, -20, 0, 28)
    nameLabel.Position = UDim2.new(0, 12, 0, 8)
    nameLabel.BackgroundTransparency = 1
    nameLabel.Text = asset.name
    nameLabel.TextColor3 = Color3.fromRGB(255, 255, 255)
    nameLabel.TextSize = 18
    nameLabel.TextXAlignment = Enum.TextXAlignment.Left
    nameLabel.Font = Enum.Font.GothamBold
    nameLabel.Parent = card
    
    local categoryBadge = Instance.new("Frame")
    categoryBadge.Size = UDim2.new(0, 0, 0, 20)
    categoryBadge.Position = UDim2.new(0, 12, 0, 38)
    categoryBadge.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
    categoryBadge.BorderSizePixel = 0
    categoryBadge.Parent = card
    
    local categoryCorner = Instance.new("UICorner")
    categoryCorner.CornerRadius = UDim.new(0, 6)
    categoryCorner.Parent = categoryBadge
    
    local categoryLabel = Instance.new("TextLabel")
    categoryLabel.Size = UDim2.new(1, -8, 1, 0)
    categoryLabel.Position = UDim2.new(0, 4, 0, 0)
    categoryLabel.BackgroundTransparency = 1
    categoryLabel.Text = asset.category or "General"
    categoryLabel.TextColor3 = Color3.fromRGB(255, 255, 255)
    categoryLabel.TextSize = 11
    categoryLabel.TextXAlignment = Enum.TextXAlignment.Left
    categoryLabel.Font = Enum.Font.GothamSemibold
    categoryLabel.Parent = categoryBadge
    
    categoryBadge.Size = UDim2.new(0, categoryLabel.TextBounds.X + 8, 0, 20)
    
    local descLabel = Instance.new("TextLabel")
    descLabel.Size = UDim2.new(1, -24, 0, 0)
    descLabel.Position = UDim2.new(0, 12, 0, 62)
    descLabel.BackgroundTransparency = 1
    descLabel.Text = asset.description and asset.description ~= "" and asset.description or "Visual effect asset for use in your game."
    descLabel.TextColor3 = Color3.fromRGB(180, 190, 220)
    descLabel.TextSize = 12
    descLabel.TextXAlignment = Enum.TextXAlignment.Left
    descLabel.TextYAlignment = Enum.TextYAlignment.Top
    descLabel.TextWrapped = true
    descLabel.Font = Enum.Font.Gotham
    descLabel.AutomaticSize = Enum.AutomaticSize.Y
    descLabel.Parent = card
    
    local assetIdContainer = Instance.new("Frame")
    assetIdContainer.Size = UDim2.new(1, -20, 0, 0)
    assetIdContainer.Position = UDim2.new(0, 10, 0, 0)
    assetIdContainer.BackgroundTransparency = 1
    assetIdContainer.Visible = false
    assetIdContainer.Parent = card
    
    local assetIdLabel = Instance.new("TextLabel")
    assetIdLabel.Size = UDim2.new(0, 70, 1, 0)
    assetIdLabel.Position = UDim2.new(0, 0, 0, 0)
    assetIdLabel.BackgroundTransparency = 1
    assetIdLabel.Text = "Asset ID: "
    assetIdLabel.TextColor3 = Color3.fromRGB(180, 190, 220)
    assetIdLabel.TextSize = 13
    assetIdLabel.TextXAlignment = Enum.TextXAlignment.Left
    assetIdLabel.Font = Enum.Font.GothamSemibold
    assetIdLabel.Parent = assetIdContainer
    
    local assetIdTextBox = Instance.new("TextBox")
    assetIdTextBox.Size = UDim2.new(1, -75, 1, 0)
    assetIdTextBox.Position = UDim2.new(0, 75, 0, 0)
    assetIdTextBox.BackgroundColor3 = Color3.fromRGB(40, 45, 85)
    assetIdTextBox.BorderSizePixel = 0
    assetIdTextBox.Text = ""
    assetIdTextBox.TextColor3 = Color3.fromRGB(100, 255, 150)
    assetIdTextBox.TextSize = 13
    assetIdTextBox.TextXAlignment = Enum.TextXAlignment.Left
    assetIdTextBox.Font = Enum.Font.GothamBold
    assetIdTextBox.ClearTextOnFocus = false
    assetIdTextBox.TextEditable = false
    assetIdTextBox.Active = true
    assetIdTextBox.Parent = assetIdContainer
    
    local assetIdCorner = Instance.new("UICorner")
    assetIdCorner.CornerRadius = UDim.new(0, 4)
    assetIdCorner.Parent = assetIdTextBox
    
    local assetIdPadding = Instance.new("UIPadding")
    assetIdPadding.PaddingLeft = UDim.new(0, 6)
    assetIdPadding.PaddingRight = UDim.new(0, 6)
    assetIdPadding.Parent = assetIdTextBox
    
    local closeButton = Instance.new("TextButton")
    closeButton.Size = UDim2.new(0, 24, 0, 24)
    closeButton.Position = UDim2.new(1, -32, 0, 8)
    closeButton.BackgroundColor3 = Color3.fromRGB(200, 50, 50)
    closeButton.BorderSizePixel = 0
    closeButton.Text = "×"
    closeButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    closeButton.TextSize = 20
    closeButton.Font = Enum.Font.GothamBold
    closeButton.Visible = false
    closeButton.ZIndex = 10
    closeButton.Parent = card
    
    local closeCorner = Instance.new("UICorner")
    closeCorner.CornerRadius = UDim.new(0, 4)
    closeCorner.Parent = closeButton
    
    local isExpanded = false
    
    local function closeCard()
        assetIdContainer.Visible = false
        closeButton.Visible = false
        isExpanded = false
        local descHeight = descLabel.TextBounds.Y > 0 and descLabel.TextBounds.Y or 20
        card.Size = UDim2.new(1, 0, 0, math.max(100, descHeight + 70))
        VFXUI.selectedId = nil
    end
    
    local function openCard()
        assetIdTextBox.Text = asset.asset_id
        assetIdContainer.Visible = true
        closeButton.Visible = true
        isExpanded = true
        assetIdContainer.Size = UDim2.new(1, -20, 0, 32)
        local descHeight = descLabel.TextBounds.Y > 0 and descLabel.TextBounds.Y or 20
        assetIdContainer.Position = UDim2.new(0, 10, 0, math.max(100, descHeight + 70))
        card.Size = UDim2.new(1, 0, 0, math.max(135, descHeight + 102))
        VFXUI.selectedId = asset.asset_id
    end
    
    closeButton.MouseButton1Click:Connect(closeCard)
    
    assetIdTextBox.Focused:Connect(function()
        task.wait(0.05)
        if assetIdTextBox.Text ~= "" then
            assetIdTextBox.SelectionStart = 1
            assetIdTextBox.CursorPosition = #assetIdTextBox.Text + 1
        end
    end)
    
    
    card:GetPropertyChangedSignal("AbsoluteSize"):Connect(function()
        if descLabel then
            descLabel.Size = UDim2.new(1, -24, 0, 0)
        end
    end)
    
    card.MouseButton1Click:Connect(function()
        if not isExpanded then
            if VFXUI.selectedId and VFXUI.selectedId ~= asset.asset_id then
                return
            end
            openCard()
        end
    end)
    
    local descHeight = descLabel.TextBounds.Y > 0 and descLabel.TextBounds.Y or 20
    card.Size = UDim2.new(1, 0, 0, math.max(100, descHeight + 70))
    
    descLabel:GetPropertyChangedSignal("TextBounds"):Connect(function()
        local newHeight = descLabel.TextBounds.Y > 0 and descLabel.TextBounds.Y or 20
        if not isExpanded then
            card.Size = UDim2.new(1, 0, 0, math.max(100, newHeight + 70))
        end
    end)
end

function VFXUI.load()
    for _, child in ipairs(VFXUI.ScrollFrame:GetChildren()) do
        if child:IsA("TextButton") or child:IsA("Frame") then
            child:Destroy()
        end
    end
    
    for _, asset in ipairs(VFXUI.assets) do
        VFXUI.createCard(asset)
    end
end

VFXPage:GetPropertyChangedSignal("Visible"):Connect(function()
    if VFXPage.Visible then
        task.wait(0.1)
        VFXUI.load()
    end
end)

-- Also load when tab is clicked

-- Reset to Defaults Button
-- Token Usage Section
SettingsUI.TokenSection = Instance.new("Frame")
SettingsUI.TokenSection.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.TokenSection.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.TokenSection.BackgroundTransparency = 1
SettingsUI.TokenSection.LayoutOrder = 6
SettingsUI.TokenSection.Parent = SettingsPage

SettingsUI.TokenSectionLayout = Instance.new("UIListLayout")
SettingsUI.TokenSectionLayout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.TokenSectionLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.TokenSectionLayout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.TokenSectionLayout.Padding = UDim.new(0, 12)
SettingsUI.TokenSectionLayout.Parent = SettingsUI.TokenSection

SettingsUI.TokenLabel = Instance.new("TextLabel")
SettingsUI.TokenLabel.Size = UDim2.new(1, 0, 0, 20)
SettingsUI.TokenLabel.BackgroundTransparency = 1
SettingsUI.TokenLabel.Text = "Token Usage"
SettingsUI.TokenLabel.TextColor3 = Color3.fromRGB(220, 230, 255)
SettingsUI.TokenLabel.TextScaled = false
SettingsUI.TokenLabel.TextSize = 14
SettingsUI.TokenLabel.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.TokenLabel.Font = Enum.Font.GothamSemibold
SettingsUI.TokenLabel.LayoutOrder = 1
SettingsUI.TokenLabel.Parent = SettingsUI.TokenSection

-- Token Usage Display Container
SettingsUI.TokenDisplay = Instance.new("Frame")
SettingsUI.TokenDisplay.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.TokenDisplay.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.TokenDisplay.BackgroundTransparency = 1
SettingsUI.TokenDisplay.LayoutOrder = 2
SettingsUI.TokenDisplay.Parent = SettingsUI.TokenSection

SettingsUI.TokenDisplayLayout = Instance.new("UIListLayout")
SettingsUI.TokenDisplayLayout.FillDirection = Enum.FillDirection.Vertical
SettingsUI.TokenDisplayLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
SettingsUI.TokenDisplayLayout.SortOrder = Enum.SortOrder.LayoutOrder
SettingsUI.TokenDisplayLayout.Padding = UDim.new(0, 8)
SettingsUI.TokenDisplayLayout.Parent = SettingsUI.TokenDisplay

-- Token Count Text
SettingsUI.TokenCount = Instance.new("TextLabel")
SettingsUI.TokenCount.Size = UDim2.new(1, 0, 0, 18)
SettingsUI.TokenCount.BackgroundTransparency = 1
SettingsUI.TokenCount.Text = "Loading..."
SettingsUI.TokenCount.TextColor3 = Color3.fromRGB(235, 240, 255)
SettingsUI.TokenCount.TextScaled = false
SettingsUI.TokenCount.TextSize = 13
SettingsUI.TokenCount.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.TokenCount.Font = Enum.Font.Gotham
SettingsUI.TokenCount.LayoutOrder = 1
SettingsUI.TokenCount.Parent = SettingsUI.TokenDisplay

-- Progress Bar Background
SettingsUI.TokenProgressBg = Instance.new("Frame")
SettingsUI.TokenProgressBg.Size = UDim2.new(1, 0, 0, 8)
SettingsUI.TokenProgressBg.BackgroundColor3 = Color3.fromRGB(30, 35, 60)
SettingsUI.TokenProgressBg.BorderSizePixel = 0
SettingsUI.TokenProgressBg.LayoutOrder = 2
SettingsUI.TokenProgressBg.Parent = SettingsUI.TokenDisplay

SettingsUI.TokenProgressBgCorner = Instance.new("UICorner")
SettingsUI.TokenProgressBgCorner.CornerRadius = UDim.new(0, 4)
SettingsUI.TokenProgressBgCorner.Parent = SettingsUI.TokenProgressBg

-- Progress Bar Fill
SettingsUI.TokenProgressFill = Instance.new("Frame")
SettingsUI.TokenProgressFill.Size = UDim2.new(0, 0, 1, 0)
SettingsUI.TokenProgressFill.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
SettingsUI.TokenProgressFill.BorderSizePixel = 0
SettingsUI.TokenProgressFill.Parent = SettingsUI.TokenProgressBg

SettingsUI.TokenProgressFillCorner = Instance.new("UICorner")
SettingsUI.TokenProgressFillCorner.CornerRadius = UDim.new(0, 4)
SettingsUI.TokenProgressFillCorner.Parent = SettingsUI.TokenProgressFill

-- Reset Date Text
SettingsUI.TokenResetDate = Instance.new("TextLabel")
SettingsUI.TokenResetDate.Size = UDim2.new(1, 0, 0, 14)
SettingsUI.TokenResetDate.BackgroundTransparency = 1
SettingsUI.TokenResetDate.Text = ""
SettingsUI.TokenResetDate.TextColor3 = Color3.fromRGB(150, 160, 200)
SettingsUI.TokenResetDate.TextScaled = false
SettingsUI.TokenResetDate.TextSize = 11
SettingsUI.TokenResetDate.TextXAlignment = Enum.TextXAlignment.Left
SettingsUI.TokenResetDate.Font = Enum.Font.Gotham
SettingsUI.TokenResetDate.LayoutOrder = 3
SettingsUI.TokenResetDate.Parent = SettingsUI.TokenDisplay

-- Function to update token usage display
local function updateTokenUsage()
    local apiKey = getActiveApiKey()
    if not apiKey or apiKey == "" then
        SettingsUI.TokenCount.Text = "Enter API key to view usage"
        SettingsUI.TokenProgressFill.Size = UDim2.new(0, 0, 1, 0)
        SettingsUI.TokenResetDate.Text = ""
        return
    end
    
    local success, response = pcall(function()
        return HttpService:GetAsync("https://electrodeai.org/api/token-usage?api_key=" .. apiKey)
    end)
    
    if success and response then
        local ok, data = pcall(function()
            return HttpService:JSONDecode(response)
        end)
        
        if ok and data and not data.error then
            local tokensUsed = data.tokens_used or 0
            local tokenLimit = data.token_limit or 2000000
            local tokensRemaining = data.tokens_remaining or (tokenLimit - tokensUsed)
            local resetDate = data.reset_date or ""
            
            -- Format numbers with commas
            local function formatNumber(num)
                local formatted = tostring(num)
                local k
                while true do
                    formatted, k = string.gsub(formatted, "^(-?%d+)(%d%d%d)", '%1,%2')
                    if k == 0 then break end
                end
                return formatted
            end
            
            SettingsUI.TokenCount.Text = formatNumber(tokensUsed) .. " / " .. formatNumber(tokenLimit) .. " tokens"
            
            -- Update progress bar
            local percentage = math.clamp(tokensUsed / tokenLimit, 0, 1)
            SettingsUI.TokenProgressFill.Size = UDim2.new(percentage, 0, 1, 0)
            
            -- Color based on usage
            if percentage >= 0.9 then
                SettingsUI.TokenProgressFill.BackgroundColor3 = Color3.fromRGB(255, 100, 100)
            elseif percentage >= 0.75 then
                SettingsUI.TokenProgressFill.BackgroundColor3 = Color3.fromRGB(255, 180, 100)
            else
                SettingsUI.TokenProgressFill.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
            end
            
            -- Update reset date
            if resetDate ~= "" then
                SettingsUI.TokenResetDate.Text = "Resets: " .. resetDate
            else
                SettingsUI.TokenResetDate.Text = ""
            end
        else
            SettingsUI.TokenCount.Text = "Unable to load token usage"
            SettingsUI.TokenProgressFill.Size = UDim2.new(0, 0, 1, 0)
            SettingsUI.TokenResetDate.Text = ""
        end
    else
        SettingsUI.TokenCount.Text = "Unable to load token usage"
        SettingsUI.TokenProgressFill.Size = UDim2.new(0, 0, 1, 0)
        SettingsUI.TokenResetDate.Text = ""
    end
end

-- Update token usage when settings tab is opened
SettingsTabButton.MouseButton1Click:Connect(function()
    activeTab = "Settings"
    updateTokenUsage()
end)

-- Initial update
task.delay(1, updateTokenUsage)

SettingsUI.ResetSection = Instance.new("Frame")
SettingsUI.ResetSection.Size = UDim2.new(1, 0, 0, 0)
SettingsUI.ResetSection.AutomaticSize = Enum.AutomaticSize.Y
SettingsUI.ResetSection.BackgroundTransparency = 1
SettingsUI.ResetSection.LayoutOrder = 7
SettingsUI.ResetSection.Parent = SettingsPage

SettingsUI.ResetButton = Instance.new("TextButton")
SettingsUI.ResetButton.Size = UDim2.new(0, 180, 0, 36)
SettingsUI.ResetButton.Text = "Reset to Defaults"
SettingsUI.ResetButton.TextColor3 = Color3.fromRGB(255, 255, 255)
SettingsUI.ResetButton.TextScaled = false
SettingsUI.ResetButton.TextSize = 13
SettingsUI.ResetButton.Font = Enum.Font.GothamSemibold
SettingsUI.ResetButton.LayoutOrder = 1
SettingsUI.ResetButton.Parent = SettingsUI.ResetSection
Style.styleSecondaryButton(SettingsUI.ResetButton)

SettingsUI.ResetButton.MouseButton1Click:Connect(function()
    -- Reset all settings except API key
    SettingsUI.AutoSaveToggle:SetAttribute("Enabled", true)
    updateAutoSaveToggle()
    pcall(function()
        plugin:SetSetting("AutoSaveChat", true)
    end)
    
    SettingsUI.ContextCardsToggle:SetAttribute("Enabled", true)
    updateContextCardsToggle()
    pcall(function()
        plugin:SetSetting("ShowContextCards", true)
    end)
    
    -- Status message removed since there's no status label in Settings anymore
end)

-- Settings page is now ready

-- Legacy global aliases removed - API key management moved to popup

-- Layout functions removed - Settings page uses UIListLayout

-- Backend URL (change this to your deployed backend URL)
local BACKEND_URL = "https://electrodeai.org"

-- Removed unused Generator variables to reduce register usage
local cachedApiKey, confirmationOverlay = nil, nil

local agentModelOptions = {
    {
        id = "planning",
        label = "Planning",
        button = AgentUI.ModelOption1
    },
    {
        id = "agent",
        label = "Agent",
        button = AgentUI.ModelOption2
    }
}

local AGENT_SESSION_KEY = "ElectrodeAgentSession_v2"

local agentState = {
    model = agentModelOptions[1].id,
    trusted = false,
    conversation = {},
    plan = {},
    planStatus = {},
    sessionId = HttpService:GenerateGUID(false),
    lastAssistantMessage = nil,
    diffEntries = {},
    alwaysAllow = false,
    liveNarrationActive = false,
    liveNarrationIndex = 1,
    pendingAssistantChunks = 0,
    justDidAutoFollowup = false -- Flag to prevent infinite query loops
}

local agentRequestInFlight = false
local saveAgentSession -- forward declaration for cross references
local captureSnapshot -- forward declaration for cross references

function Style.setModelButtonAppearance(button, active)
    if not button then
        return
    end

    local gradient = button:FindFirstChild("ButtonGradient")
    if gradient then
        gradient.Color = active and ColorSequence.new({
            ColorSequenceKeypoint.new(0, Color3.fromRGB(170, 140, 255)),
            ColorSequenceKeypoint.new(1, Color3.fromRGB(95, 125, 255))
        }) or ColorSequence.new({
            ColorSequenceKeypoint.new(0, Color3.fromRGB(46, 55, 110)),
            ColorSequenceKeypoint.new(1, Color3.fromRGB(28, 32, 72))
        })
        gradient.Transparency = active and NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.05),
            NumberSequenceKeypoint.new(1, 0.2)
        }) or NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.18),
            NumberSequenceKeypoint.new(1, 0.4)
        })
    end

    local stroke = button:FindFirstChild("ButtonStroke")
    if stroke then
        stroke.Color = active and Color3.fromRGB(190, 170, 255) or Color3.fromRGB(120, 150, 255)
        stroke.Transparency = active and 0.15 or 0.4
    end

    button.BackgroundTransparency = active and 0.08 or 0.2
    button.TextColor3 = Color3.fromRGB(255, 255, 255)
end

function Style.setTrustedToggleAppearance(button, enabled)
    if not button then
        return
    end

    local gradient = button:FindFirstChild("ButtonGradient")
    if gradient then
        gradient.Color = enabled and ColorSequence.new({
            ColorSequenceKeypoint.new(0, Color3.fromRGB(92, 204, 142)),
            ColorSequenceKeypoint.new(1, Color3.fromRGB(42, 168, 98))
        }) or ColorSequence.new({
            ColorSequenceKeypoint.new(0, Color3.fromRGB(46, 55, 110)),
            ColorSequenceKeypoint.new(1, Color3.fromRGB(28, 32, 72))
        })
        gradient.Transparency = enabled and NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.05),
            NumberSequenceKeypoint.new(1, 0.2)
        }) or NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.18),
            NumberSequenceKeypoint.new(1, 0.4)
        })
    end

    local stroke = button:FindFirstChild("ButtonStroke")
    if stroke then
        stroke.Color = enabled and Color3.fromRGB(144, 220, 180) or Color3.fromRGB(120, 150, 255)
        stroke.Transparency = enabled and 0.15 or 0.4
    end

    button.BackgroundTransparency = enabled and 0.08 or 0.2
    button.TextColor3 = Color3.fromRGB(255, 255, 255)
end
function Network.getRobloxUserId()
    local success, userId = pcall(function()
        return game:GetService("StudioService"):GetUserId()
    end)
    
    if success and userId > 0 then
        return tostring(userId)
    else
        return nil
    end
end
local updatePlanUI -- forward declaration

local snapshotState = {
    current = nil,
    previous = nil,
    dirty = true,
    capturing = false,
    scheduled = false,
    callbacks = {},
    lastReason = "startup",
    diff = {}
}

-- Error detection state (minimal locals)
local errorState = {
    errors = {},
    isPlaytesting = false,
    currentContinueToken = nil, -- Store continue token for seamless query continuation
    queryContinuationPending = false, -- Flag to track if query continuation is in progress
    queryContinuationDone = false, -- Flag to prevent infinite query loops - tracks if we've already done one continuation
    queryOnlyContinuationCount = 0, -- Counter for query-only continuations (prevents infinite loops)
    monitorConnection = nil, -- Connection for real-time error monitoring
    logServiceConnection = nil, -- Connection for LogService.MessageOut events
    useEventBasedDetection = false, -- Whether event-based detection is working
    pendingErrorRequest = false, -- Flag to track if error request is queued for when test stops
    pendingErrorMessages = {}, -- Queued error messages to send when test stops
    lastErrorCheckTime = nil -- Timestamp of last error check for throttling
}

-- Attach to agentState to avoid module-level locals
agentState.setModel = function(self, modelId, suppressSave)
    for _, option in ipairs(agentModelOptions) do
        local isSelected = option.id == modelId
        Style.setModelButtonAppearance(option.button, isSelected)
    end
    self.model = modelId
    if not suppressSave then
        pcall(function()
            plugin:SetSetting("AgentModel", modelId)
        end)
        saveAgentSession()
    end
end
local function setAgentModel(modelId, suppressSave)
    return agentState:setModel(modelId, suppressSave)
end

agentState.setTrustedMode = function(self, enabled)
    self.trusted = enabled
    AgentUI.TrustToggle.Text = enabled and "Auto Apply Changes" or "Require Approval"
    Style.setTrustedToggleAppearance(AgentUI.TrustToggle, enabled)
    pcall(function()
        plugin:SetSetting("AgentTrustedMode", enabled)
    end)
    saveAgentSession()
end
local function setTrustedMode(enabled)
    return agentState:setTrustedMode(enabled)
end

local function clearAgentChat()
    -- Don't clear if a request is in flight
    if agentRequestInFlight then
        appendConversationMessage("assistant", "Can't clear chat while I'm working. Wait for me to finish first.")
        return
    end
    
    -- Clear all chat bubbles (except ThoughtWrapper)
    for _, child in ipairs(AgentUI.ChatScroll:GetChildren()) do
        if child:IsA("Frame") and child ~= AgentUI.ThoughtWrapper then
            child:Destroy()
        end
    end
    
    -- Reset conversation state
    agentState.conversation = {}
    agentState.plan = {}
    agentState.planStatus = {}
    agentState.sessionId = HttpService:GenerateGUID(false)
    agentState.lastAssistantMessage = nil
    agentState.diffEntries = {}
    chatBubbleCounter = 0
    
    -- Hide any thought stages
    hideThoughtStages()
    
    -- Clear the plan UI visually
    if updatePlanUI then
        updatePlanUI({})
    end
    
    -- Save the cleared state
    if saveAgentSession then
        saveAgentSession()
    end
    
    -- Reset canvas position
    AgentUI.ChatScroll.CanvasPosition = Vector2.new(0, 0)
end

AgentModelOption1.MouseButton1Click:Connect(function()
    setAgentModel(agentModelOptions[1].id)
end)

AgentModelOption2.MouseButton1Click:Connect(function()
    setAgentModel(agentModelOptions[2].id)
end)

AgentUI.TrustToggle.MouseButton1Click:Connect(function()
    setTrustedMode(not agentState.trusted)
end)

AgentUI.ClearChatButton.MouseButton1Click:Connect(function()
    clearAgentChat()
end)

local chatBubbleCounter = 0

-- Markdown to RichText converter
local function markdownToRichText(text)
    if not text or text == "" then
        return text
    end
    
    -- Detect if content is JSON or raw debug output - skip markdown conversion
    -- Check for raw_response patterns or JSON structure
    local trimmedText = text:gsub("^%s+", ""):gsub("%s+$", "")
    local isRawResponse = text:find("Workspace Query Results") or text:find("Raw model response")
    local looksLikeJson = (trimmedText:sub(1, 1) == "{" or trimmedText:sub(1, 1) == "[") and 
                          (text:find('"assistant_message"') or text:find('"plan"') or text:find('"actions"') or
                           text:find('"type"') or text:find('"path"') or text:find('"schema_failed"'))
    
    -- If it's raw response or JSON, preserve as code, no markdown processing
    if isRawResponse or looksLikeJson then
        -- Format as code block but preserve original structure
        local formattedJson = text:gsub("<", "&lt;"):gsub(">", "&gt;")
        return '<font face="Code">' .. formattedJson:gsub("\n", "<br/>") .. '</font>'
    end
    
    local result = text
    
    -- Store code blocks first to prevent them from being processed
    local codeBlocks = {}
    local codeBlockIndex = 0
    
    -- Code blocks (```code```) - handle multiline
    result = result:gsub("```([^`\n]-)```", function(code)
        codeBlockIndex = codeBlockIndex + 1
        local placeholder = "___CODE_BLOCK_" .. codeBlockIndex .. "___"
        codeBlocks[placeholder] = code
        return placeholder
    end)
    
    -- Also handle multiline code blocks
    result = result:gsub("```([^`]+)```", function(code)
        codeBlockIndex = codeBlockIndex + 1
        local placeholder = "___CODE_BLOCK_" .. codeBlockIndex .. "___"
        codeBlocks[placeholder] = code
        return placeholder
    end)
    
    -- Escape existing RichText tags to prevent conflicts (but not our placeholders)
    result = result:gsub("<([^/])", "&lt;%1")
    result = result:gsub("</", "&lt;/")
    result = result:gsub(">", "&gt;")
    
    -- Inline code (`code`) - but not inside code blocks
    result = result:gsub("`([^`]+)`", function(code)
        return '<font face="Code">' .. code .. '</font>'
    end)
    
    -- Process line by line for headers, lists, and blockquotes
    local lines = {}
    for line in result:gmatch("[^\n]+") do
        -- Headers (use bold for emphasis, RichText doesn't support font size changes easily)
        if line:match("^###%s+") then
            line = line:gsub("^###%s+(.+)$", "<b>%1</b>")
        elseif line:match("^##%s+") then
            line = line:gsub("^##%s+(.+)$", "<b>%1</b>")
        elseif line:match("^#%s+") then
            line = line:gsub("^#%s+(.+)$", "<b>%1</b>")
        -- Lists
        elseif line:match("^%s*[-*]%s+") then
            line = line:gsub("^%s*[-*]%s+(.+)$", "• %1")
        -- Blockquotes
        elseif line:match("^>%s+") then
            line = line:gsub("^>%s+(.+)$", "<i>%1</i>")
        end
        table.insert(lines, line)
    end
    result = table.concat(lines, "\n")
    
    -- Bold (**text** or __text__) - process before italic
    result = result:gsub("%*%*([^*]+)%*%*", "<b>%1</b>")
    result = result:gsub("__([^_]+)__", "<b>%1</b>")
    
    -- Italic (*text* or _text_) - but not if already bold
    result = result:gsub("%*([^*]+)%*", function(text)
        if not text:find("<b>") and not text:find("</b>") then
            return "<i>" .. text .. "</i>"
        end
        return "*" .. text .. "*"
    end)
    result = result:gsub("_([^_]+)_", function(text)
        if not text:find("<b>") and not text:find("</b>") then
            return "<i>" .. text .. "</i>"
        end
        return "_" .. text .. "_"
    end)
    
    -- Links [text](url) - RichText doesn't support clickable links, so just show text
    result = result:gsub("%[([^%]]+)%]%([^%)]+%)", "%1")
    
    -- Line breaks
    result = result:gsub("\n", "<br/>")
    
    -- Restore code blocks
    for placeholder, code in pairs(codeBlocks) do
        local formattedCode = code:gsub("\n", "<br/>")
        formattedCode = formattedCode:gsub("&lt;", "<")
        formattedCode = formattedCode:gsub("&gt;", ">")
        result = result:gsub(placeholder, '<font face="Code">' .. formattedCode .. '</font>')
    end
    
    -- Restore escaped characters (but not in code blocks)
    result = result:gsub("&lt;", "<")
    result = result:gsub("&gt;", ">")
    
    return result
end

local function createChatBubble(message)
    chatBubbleCounter = chatBubbleCounter + 1
    local container = Instance.new("Frame")
    container.BackgroundTransparency = 1
    container.Size = UDim2.new(1, 0, 0, 0)
    container.AutomaticSize = Enum.AutomaticSize.Y
    container.LayoutOrder = chatBubbleCounter
    container.Parent = AgentUI.ChatScroll

    local bubblePadding = Instance.new("UIPadding")
    bubblePadding.PaddingTop = UDim.new(0, 0)
    bubblePadding.PaddingBottom = UDim.new(0, 10)
    if message.role == "user" then
        bubblePadding.PaddingLeft = UDim.new(0, 80)
        bubblePadding.PaddingRight = UDim.new(0, 12)
    else
        bubblePadding.PaddingLeft = UDim.new(0, 12)
        bubblePadding.PaddingRight = UDim.new(0, 80)
    end
    bubblePadding.Parent = container

    local bubble = Instance.new("Frame")
    bubble.Name = message.role == "user" and "UserBubble" or "AgentBubble"
    bubble.AutomaticSize = Enum.AutomaticSize.Y
    bubble.Size = UDim2.new(1, 0, 0, 0)
    bubble.BackgroundColor3 = message.role == "user" and Color3.fromRGB(102, 126, 234) or Color3.fromRGB(236, 240, 243)
    bubble.BorderSizePixel = 0
    bubble.Parent = container

    local bubbleCorner = Instance.new("UICorner")
    bubbleCorner.CornerRadius = UDim.new(0, 10)
    bubbleCorner.Parent = bubble

    local bubblePaddingInner = Instance.new("UIPadding")
    bubblePaddingInner.PaddingTop = UDim.new(0, 10)
    bubblePaddingInner.PaddingBottom = UDim.new(0, 10)
    bubblePaddingInner.PaddingLeft = UDim.new(0, 12)
    bubblePaddingInner.PaddingRight = UDim.new(0, 12)
    bubblePaddingInner.Parent = bubble

    local contentText = message.content or ""
    -- Check for workspace query results in both assistant and system messages
    local isWorkspaceQuery = (
        (message.role == "assistant" or message.role == "system") and (
            contentText:find("Workspace Query Results") or
            contentText:find("workspace") or
            contentText:find("Workspace") or
            contentText:find("query") or
            contentText:find("Raw model response") or
            contentText:len() > 500
        )
    )
    
    -- Check if content is long enough to collapse
    local shouldCollapse = isWorkspaceQuery and contentText:len() > 300
    
    if shouldCollapse then
        -- Create collapsible container
        local collapsibleContainer = Instance.new("Frame")
        collapsibleContainer.BackgroundTransparency = 1
        collapsibleContainer.Size = UDim2.new(1, 0, 0, 0)
        collapsibleContainer.AutomaticSize = Enum.AutomaticSize.Y
        collapsibleContainer.Parent = bubble
        
        local collapsibleLayout = Instance.new("UIListLayout")
        collapsibleLayout.FillDirection = Enum.FillDirection.Vertical
        collapsibleLayout.SortOrder = Enum.SortOrder.LayoutOrder
        collapsibleLayout.Padding = UDim.new(0, 8)
        collapsibleLayout.Parent = collapsibleContainer
        
        -- Preview text (first 200 chars)
        local previewText = contentText:sub(1, 200) .. "..."
        local previewLabel = Instance.new("TextLabel")
        previewLabel.BackgroundTransparency = 1
        previewLabel.Size = UDim2.new(1, 0, 0, 0)
        previewLabel.AutomaticSize = Enum.AutomaticSize.Y
        previewLabel.Font = Enum.Font.SourceSans
        previewLabel.TextSize = 15
        previewLabel.TextWrapped = true
        previewLabel.TextXAlignment = Enum.TextXAlignment.Left
        previewLabel.TextYAlignment = Enum.TextYAlignment.Top
        previewLabel.TextColor3 = Color3.fromRGB(44, 62, 80)
        previewLabel.RichText = true
        previewLabel.Text = markdownToRichText(previewText)
        previewLabel.LayoutOrder = 1
        previewLabel.Parent = collapsibleContainer
        
        -- Expand/Collapse button
        local expandButton = Instance.new("TextButton")
        expandButton.Size = UDim2.new(0, 120, 0, 24)
        expandButton.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
        expandButton.Text = "▼ Expand"
        expandButton.TextColor3 = Color3.fromRGB(255, 255, 255)
        expandButton.TextSize = 12
        expandButton.Font = Enum.Font.GothamSemibold
        expandButton.LayoutOrder = 2
        expandButton.Parent = collapsibleContainer
        
        local expandButtonCorner = Instance.new("UICorner")
        expandButtonCorner.CornerRadius = UDim.new(0, 6)
        expandButtonCorner.Parent = expandButton
        
        -- Full content (hidden by default)
        local fullContentLabel = Instance.new("TextLabel")
        fullContentLabel.BackgroundTransparency = 1
        fullContentLabel.Size = UDim2.new(1, 0, 0, 0)
        fullContentLabel.AutomaticSize = Enum.AutomaticSize.Y
        fullContentLabel.Font = Enum.Font.SourceSans
        fullContentLabel.TextSize = 13
        fullContentLabel.TextWrapped = true
        fullContentLabel.TextXAlignment = Enum.TextXAlignment.Left
        fullContentLabel.TextYAlignment = Enum.TextYAlignment.Top
        fullContentLabel.TextColor3 = Color3.fromRGB(60, 70, 90)
        fullContentLabel.RichText = true
        fullContentLabel.Text = markdownToRichText(contentText)
        fullContentLabel.Visible = false
        fullContentLabel.LayoutOrder = 3
        fullContentLabel.Parent = collapsibleContainer
        
        -- Collapse button (hidden by default)
        local collapseButton = Instance.new("TextButton")
        collapseButton.Size = UDim2.new(0, 120, 0, 24)
        collapseButton.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
        collapseButton.Text = "▲ Collapse"
        collapseButton.TextColor3 = Color3.fromRGB(255, 255, 255)
        collapseButton.TextSize = 12
        collapseButton.Font = Enum.Font.GothamSemibold
        collapseButton.Visible = false
        collapseButton.LayoutOrder = 4
        collapseButton.Parent = collapsibleContainer
        
        local collapseButtonCorner = Instance.new("UICorner")
        collapseButtonCorner.CornerRadius = UDim.new(0, 6)
        collapseButtonCorner.Parent = collapseButton
        
        -- Toggle functionality
        local isExpanded = false
        expandButton.MouseButton1Click:Connect(function()
            isExpanded = true
            previewLabel.Visible = false
            fullContentLabel.Visible = true
            expandButton.Visible = false
            collapseButton.Visible = true
        end)
        
        collapseButton.MouseButton1Click:Connect(function()
            isExpanded = false
            previewLabel.Visible = true
            fullContentLabel.Visible = false
            expandButton.Visible = true
            collapseButton.Visible = false
            -- Scroll to top of container
            task.wait()
            AgentUI.ChatScroll.CanvasPosition = Vector2.new(0, container.AbsolutePosition.Y - AgentUI.ChatScroll.AbsolutePosition.Y)
        end)
    else
        -- Regular non-collapsible content
        local contentLabel = Instance.new("TextLabel")
        contentLabel.BackgroundTransparency = 1
        contentLabel.Size = UDim2.new(1, 0, 0, 0)
        contentLabel.AutomaticSize = Enum.AutomaticSize.Y
        contentLabel.Font = Enum.Font.SourceSans
        contentLabel.TextSize = 15
        contentLabel.TextWrapped = true
        contentLabel.TextXAlignment = Enum.TextXAlignment.Left
        contentLabel.TextYAlignment = Enum.TextYAlignment.Top
        contentLabel.TextColor3 = message.role == "user" and Color3.fromRGB(255, 255, 255) or Color3.fromRGB(44, 62, 80)
        contentLabel.RichText = true
        contentLabel.Text = markdownToRichText(contentText)
        contentLabel.Parent = bubble
    end

    if message.notes then
        local notesLabel = Instance.new("TextLabel")
        notesLabel.BackgroundTransparency = 1
        notesLabel.Size = UDim2.new(1, 0, 0, 14)
        notesLabel.Position = UDim2.new(0, 0, 1, 4)
        notesLabel.AutomaticSize = Enum.AutomaticSize.Y
        notesLabel.Font = Enum.Font.SourceSansItalic
        notesLabel.TextSize = 12
        notesLabel.TextWrapped = true
        notesLabel.TextXAlignment = Enum.TextXAlignment.Left
        notesLabel.TextColor3 = message.role == "user" and Color3.fromRGB(220, 230, 255) or Color3.fromRGB(104, 113, 122)
        notesLabel.Text = message.notes
        notesLabel.Parent = bubble
    end

    return container
end

-- Remove message limit - allow infinite scrolling
-- local MAX_AGENT_MESSAGES = 60 -- Disabled: allow infinite chat history

local function pruneConversation()
    -- Disabled: No longer pruning messages to allow infinite scrolling
    -- Users can manually clear chat if needed
    -- while #agentState.conversation > MAX_AGENT_MESSAGES do
    --     table.remove(agentState.conversation, 1)
    --     local oldest = nil
    --     for _, child in ipairs(AgentUI.ChatScroll:GetChildren()) do
    --         if child:IsA("Frame") then
    --             oldest = child
    --             break
    --         end
    --     end
    --     if oldest then
    --         oldest:Destroy()
    --     end
    -- end
end

local function formatAssistantNarration(text)
    if typeof(text) ~= "string" or text == "" then
        return text
    end

    local sectionOrder = { "observations", "plan", "actions", "status" }
    local parsed = {}
    local currentKey = nil

    for line in text:gmatch("[^\r\n]+") do
        local trimmed = line:match("^%s*(.-)%s*$")
        if trimmed ~= "" then
            local lower = trimmed:lower()
            local foundKey = nil
            for _, key in ipairs(sectionOrder) do
                if lower:match("^" .. key) then
                    foundKey = key
                    break
                end
            end

            if foundKey then
                currentKey = foundKey
                parsed[currentKey] = parsed[currentKey] or {}
            elseif currentKey then
                local bullet = trimmed:gsub("^[-•%s]+", "")
                if bullet ~= "" then
                    table.insert(parsed[currentKey], bullet)
                end
            end
        end
    end

    local function entriesToSentence(entries)
        local count = #entries
        if count == 1 then
            return entries[1]
        end

        local buffer = {}
        for index, entry in ipairs(entries) do
            if index == 1 then
                table.insert(buffer, entry)
            elseif index == count then
                table.insert(buffer, "and " .. entry)
            else
                table.insert(buffer, entry)
            end
        end
        return table.concat(buffer, count > 2 and ", " or " ")
    end

    local segments = {}
    if parsed.plan and #parsed.plan > 0 then
        table.insert(segments, "Plan of attack: " .. entriesToSentence(parsed.plan) .. ".")
    end
    if parsed.actions and #parsed.actions > 0 then
        table.insert(segments, "Doing now: " .. entriesToSentence(parsed.actions) .. ".")
    end
    if parsed.status and #parsed.status > 0 then
        table.insert(segments, "Next up: " .. entriesToSentence(parsed.status) .. ".")
    end

    if #segments > 0 then
        return segments
    end

    return text
end

appendConversationMessage = function(role, content, extra, skipFormatting)
    local enableTypingIndicator = agentRequestInFlight and extra and extra.enableTypingIndicator
    if role == "assistant" and not skipFormatting then
        local formatted = formatAssistantNarration(content)
        if typeof(formatted) == "table" then
            if #formatted > 0 then
                if enableTypingIndicator then
                    agentState.pendingAssistantChunks = #formatted
                    showTypingIndicator()
                end
                for index, chunk in ipairs(formatted) do
                    local delaySeconds = (index - 1) * 0.45
                    local chunkExtra = nil
                    if enableTypingIndicator then
                        chunkExtra = index == 1 and extra or { enableTypingIndicator = true }
                    elseif index == 1 then
                        chunkExtra = extra
                    end
                    task.delay(delaySeconds, function()
                        appendConversationMessage(role, chunk, chunkExtra, true)
                    end)
                end
            end
            return
        elseif formatted then
            content = formatted
            if enableTypingIndicator then
                agentState.pendingAssistantChunks = 1
                showTypingIndicator()
            end
        end
    end
    local message = {
        role = role,
        content = content,
        timestamp = os.time(),
        notes = extra and extra.notes or nil
    }
    table.insert(agentState.conversation, message)
    createChatBubble(message)
    pruneConversation()
    if role == "assistant" then
        if enableTypingIndicator and agentState.pendingAssistantChunks and agentState.pendingAssistantChunks > 0 then
            agentState.pendingAssistantChunks = math.max(agentState.pendingAssistantChunks - 1, 0)
            hideThoughtStages()
            if agentState.pendingAssistantChunks > 0 then
                task.delay(0.05, function()
                    if agentRequestInFlight then
                        showTypingIndicator()
                    end
                end)
            end
        else
            hideThoughtStages()
        end
    end
end

local function stopLiveNarration()
    agentState.liveNarrationActive = false
    agentState.liveNarrationThread = nil
end

local function startLiveNarration(initialDelay)
    if agentState.liveNarrationActive then
        return
    end

    agentState.liveNarrationActive = true
    local usedBeats = {}
    local function getRandomBeat()
        if #usedBeats >= #NarrationConfig.beats then
            usedBeats = {}
        end
        local available = {}
        for i, beat in ipairs(NarrationConfig.beats) do
            local alreadyUsed = false
            for _, used in ipairs(usedBeats) do
                if used == i then
                    alreadyUsed = true
                    break
                end
            end
            if not alreadyUsed then
                table.insert(available, i)
            end
        end
        if #available == 0 then
            usedBeats = {}
            for i = 1, #NarrationConfig.beats do
                table.insert(available, i)
            end
        end
        local selectedIndex = available[math.random(1, #available)]
        table.insert(usedBeats, selectedIndex)
        return NarrationConfig.beats[selectedIndex]
    end

    agentState.liveNarrationThread = task.spawn(function()
        task.wait(initialDelay or NarrationConfig.initialDelay)
        while agentState.liveNarrationActive do
            local beat = getRandomBeat()
            if beat then
                appendConversationMessage("assistant", beat, nil, true)
            end
            local interval = NarrationConfig.minInterval + math.random() * (NarrationConfig.maxInterval - NarrationConfig.minInterval)
            task.wait(interval)
        end
    end)
end

local function describeActionHeadline(action)
    if not action or typeof(action) ~= "table" then
        return nil
    end
    if action.type == "query_workspace" then
        return string.format("search workspace for '%s'", action.query or "?")
    elseif action.type == "write_script" then
        return string.format("write %s at %s", action.class or "Script", action.path or "?")
    elseif action.type == "ensure_instance" then
        return string.format("set up %s at %s", action.class or "Folder", action.path or "?")
    elseif action.type == "set_property" then
        if action.property then
            return string.format("tune %s.%s", action.path or "?", action.property)
        elseif action.properties then
            return string.format("tune several properties on %s", action.path or "?")
        end
    elseif action.type == "delete_instance" then
        return string.format("delete %s", action.path or "?")
    elseif action.type == "rename_instance" then
        return string.format("rename %s to '%s'", action.path or "?", action.new_name or "?")
    elseif action.type == "move_instance" then
        return string.format("move %s to %s", action.path or "?", action.new_parent or "?")
    elseif action.type == "clone_instance" then
        return string.format("clone %s", action.path or "?")
    elseif action.type == "set_tags" then
        return string.format("update tags on %s", action.path or "?")
    end
    return string.format("run %s", tostring(action.type))
end

local function announceActionKickoff(actions, batchInfo, isContinuation)
    if not actions or #actions == 0 then
        return
    end
    local headline = describeActionHeadline(actions[1])
    local total = #actions
    local batchLabel = ""
    if batchInfo and batchInfo.total and batchInfo.total > 1 then
        local currentBatch = batchInfo.batch or (isContinuation and batchInfo.total or 1)
        batchLabel = string.format(" batch %d/%d", currentBatch, batchInfo.total)
    end

    local intro = isContinuation and "Picking up" or "Starting"
    local detail = headline and (" First up, I'll " .. headline) or ""
    local plurality = total > 1 and string.format(" on %d changes", total) or ""
    appendConversationMessage("assistant", string.format("%s%s%s.%s", intro, batchLabel, plurality, detail), nil, true)
end

local function clearScrollingFrame(frame)
    for _, child in ipairs(frame:GetChildren()) do
        if not child:IsA("UIListLayout") then
            child:Destroy()
        end
    end
end

-- Function to validate and filter plan entries (remove error logs)
local function validatePlanEntries(plan)
    if not plan or type(plan) ~= "table" then
        return {}
    end
    
    local validated = {}
    local errorPatterns = {
        "Response has 0 actions",
        "%[Electrode%]",
        "Error:",
        "error:",
        "failed",
        "Failed",
        "exception",
        "Exception",
        "traceback",
        "Traceback",
    }
    
    for index, step in ipairs(plan) do
        -- Skip non-table entries
        if type(step) ~= "table" then
            continue
        end
        
        local title = tostring(step.title or step.id or "Step " .. index):gsub("^%s+", ""):gsub("%s+$", "")
        local details = tostring(step.details or ""):gsub("^%s+", ""):gsub("%s+$", "")
        local combinedText = (title .. " " .. details):lower()
        
        -- Check if this looks like an error log
        local isErrorLog = false
        for _, pattern in ipairs(errorPatterns) do
            if combinedText:find(pattern:lower(), 1, true) then
                isErrorLog = true
                break
            end
        end
        
        -- Skip error logs
        if isErrorLog then
            continue
        end
        
        -- Skip entries with extremely long text (likely error logs)
        if #title > 500 or #details > 2000 then
            continue
        end
        
        -- Skip entries that look like log prefixes (e.g., "[13:59:02.448]")
        if title:match("^%[%d%d:%d%d:%d%d%.%d%d%d%]") then
            continue
        end
        
        -- Valid plan entry
        table.insert(validated, step)
    end
    
    return validated
end

updatePlanUI = function(plan)
    clearScrollingFrame(AgentUI.PlannerScroll)

    local theme = getActiveTheme()
    
    -- Validate and filter the plan first
    plan = validatePlanEntries(plan)

    if not plan or #plan == 0 then
        local label = Instance.new("TextLabel")
        label.BackgroundTransparency = 1
        label.Size = UDim2.new(1, 0, 0, 24)
        label.TextXAlignment = Enum.TextXAlignment.Left
        label.Font = Enum.Font.SourceSans
        label.TextSize = 13
        label.TextColor3 = theme.subtext
        label.Text = "No active plan. Ask electrode for a feature to begin."
        label.Parent = AgentUI.PlannerScroll
        agentState.plan = {}
        task.defer(updatePlannerCanvasSize)
        return
    end

    for index, step in ipairs(plan) do
        local row = Instance.new("Frame")
        row.Name = "PlanStep_" .. (step.id or tostring(index))
        row.Size = UDim2.new(1, 0, 0, 0)
        row.AutomaticSize = Enum.AutomaticSize.Y
        row.BackgroundColor3 = theme.cardBackground
        row.BackgroundTransparency = currentTheme == "dark" and 0.12 or 0
        row.BorderSizePixel = 0
        row.Parent = AgentUI.PlannerScroll
        row:SetAttribute("status", step.status or "pending")

        local rowLayout = Instance.new("UIListLayout")
        rowLayout.FillDirection = Enum.FillDirection.Vertical
        rowLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
        rowLayout.SortOrder = Enum.SortOrder.LayoutOrder
        rowLayout.Padding = UDim.new(0, 6)
        rowLayout.Parent = row

        local corner = Instance.new("UICorner")
        corner.CornerRadius = UDim.new(0, 6)
        corner.Parent = row

        local stroke = Instance.new("UIStroke")
        stroke.Thickness = 1
        stroke.Color = theme.cardStroke
        stroke.ApplyStrokeMode = Enum.ApplyStrokeMode.Border
        stroke.Parent = row

        local padding = Instance.new("UIPadding")
        padding.PaddingTop = UDim.new(0, 8)
        padding.PaddingBottom = UDim.new(0, 8)
        padding.PaddingLeft = UDim.new(0, 10)
        padding.PaddingRight = UDim.new(0, 10)
        padding.Parent = row

        local header = Instance.new("Frame")
        header.BackgroundTransparency = 1
        header.Size = UDim2.new(1, 0, 0, 0)
        header.AutomaticSize = Enum.AutomaticSize.Y
        header.LayoutOrder = 1
        header.Parent = row

        local headerLayout = Instance.new("UIListLayout")
        headerLayout.FillDirection = Enum.FillDirection.Horizontal
        headerLayout.HorizontalAlignment = Enum.HorizontalAlignment.Left
        headerLayout.VerticalAlignment = Enum.VerticalAlignment.Center
        headerLayout.Padding = UDim.new(0, 8)
        headerLayout.Parent = header

        local statusBadge = Instance.new("TextLabel")
        statusBadge.Name = "StatusContainer"
        statusBadge.BackgroundColor3 = Color3.fromRGB(223, 227, 235)
        statusBadge.TextColor3 = Color3.fromRGB(73, 80, 87)
        statusBadge.Font = Enum.Font.SourceSansBold
        statusBadge.TextSize = 12
        statusBadge.AutoLocalize = false
        statusBadge.Text = string.upper(step.status or "pending")
        statusBadge.AutomaticSize = Enum.AutomaticSize.XY
        statusBadge.LayoutOrder = 1
        statusBadge.Parent = header

        local badgeCorner = Instance.new("UICorner")
        badgeCorner.CornerRadius = UDim.new(1, 0)
        badgeCorner.Parent = statusBadge

        local badgePadding = Instance.new("UIPadding")
        badgePadding.PaddingTop = UDim.new(0, 2)
        badgePadding.PaddingBottom = UDim.new(0, 2)
        badgePadding.PaddingLeft = UDim.new(0, 8)
        badgePadding.PaddingRight = UDim.new(0, 8)
        badgePadding.Parent = statusBadge

        local badgeBg, badgeFg = Style.getStatusBadgeColors(step.status)
        statusBadge.BackgroundColor3 = badgeBg
        statusBadge.TextColor3 = badgeFg

        local title = Instance.new("TextLabel")
        title.Name = "TitleLabel"
        title.BackgroundTransparency = 1
        title.Size = UDim2.new(1, 0, 0, 0)
        title.AutomaticSize = Enum.AutomaticSize.Y
        title.Font = Enum.Font.SourceSansSemibold
        title.TextSize = 14
        title.TextWrapped = true
        title.TextXAlignment = Enum.TextXAlignment.Left
        title.TextColor3 = theme.text
        title.Text = step.title or "Unnamed task"
        title.LayoutOrder = 2
        title.Parent = row

        if step.details and step.details ~= "" then
            local detail = Instance.new("TextLabel")
            detail.Name = "DetailLabel"
            detail.BackgroundTransparency = 1
            detail.Size = UDim2.new(1, 0, 0, 0)
            detail.AutomaticSize = Enum.AutomaticSize.Y
            detail.Font = Enum.Font.SourceSans
            detail.TextSize = 13
            detail.TextWrapped = true
            detail.TextXAlignment = Enum.TextXAlignment.Left
            detail.TextColor3 = theme.subtext
            detail.Text = step.details
            detail.LayoutOrder = 3
            detail.Parent = row
        end
        applyPlanRowTheme(row, step.status)
    end

    agentState.plan = plan
    task.defer(updatePlannerCanvasSize)
end

local function clearChatUI()
    for _, child in ipairs(AgentUI.ChatScroll:GetChildren()) do
        if child:IsA("Frame") then
            child:Destroy()
        end
    end
end

local function reloadChatFromState()
    clearChatUI()
    for _, message in ipairs(agentState.conversation) do
        createChatBubble(message)
    end
    task.defer(updateChatCanvasSize)
end

local function resetSendButton()
    AgentUI.SendButton.Text = "Send"
    AgentUI.SendButton.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
    AgentUI.SendButton.Active = true
end

local function finalizeAgentRequest()
    resetSendButton()
    agentRequestInFlight = false
    agentState.pendingAssistantChunks = 0
    stopLiveNarration()
    task.delay(0.4, hideThoughtStages)
    
    -- Update token usage display if settings tab is active
    if activeTab == "Settings" then
        task.delay(0.5, updateTokenUsage)
    end
end

local function handleResponse(response, httpError, isContinuation)
    local apiKey = getActiveApiKey()
    -- Don't stop narration for continuations - keep it seamless
    if not isContinuation then
        stopLiveNarration()
    end
    -- For query continuations, keep UI state continuous - don't reset thought stages
    -- The thought stages should already be visible from the original request
    if httpError then
        local message = "I couldn't reach the server. Check your connection and try again."

        if typeof(httpError) == "table" then
            local body = httpError.body
            if typeof(body) == "table" then
                message = body.error or body.detail or body.message or message
            elseif typeof(body) == "string" and body ~= "" then
                message = body
            elseif httpError.statusCode then
                message = string.format("Request failed (HTTP %s).", tostring(httpError.statusCode))
            end
        else
            message = tostring(httpError)
        end

        appendConversationMessage("assistant", message)
        saveAgentSession()
        hideThoughtStages()
        finalizeAgentRequest()
        return
    end

    if not response then
        appendConversationMessage("assistant", "I couldn't reach the server. Check your connection and try again.")
        saveAgentSession()
        hideThoughtStages()
        finalizeAgentRequest()
        return
    end

    if response.error or response.detail or response.success == false then
        local message = response.error or response.detail or response.message or "The server could not process the request."
        appendConversationMessage("assistant", message)
        saveAgentSession()
        hideThoughtStages()
        finalizeAgentRequest()
        return
    end

    -- For continuations (especially query continuations), don't reset thought stage
    -- Keep the UI state continuous - only update to stage 4 if not a continuation
    if not isContinuation then
        setThoughtStage(4)
    end

    -- Debug: Log response structure to help diagnose issues
    if response.actions then
        if #response.actions == 0 then
            print("[Electrode] Response has 0 actions (empty array)")
            if response.schema_failed then
                print("[Electrode] Schema failed - AI didn't follow JSON format")
            end
            if response.error then
                print("[Electrode] Response error:", response.error)
            end
            if response.assistant_message then
                print("[Electrode] Assistant message:", response.assistant_message:sub(1, 200))
            end
        else
            print("[Electrode] Response has", #response.actions, "actions")
        end
    else
        print("[Electrode] Response has NO actions (null/undefined)")
        if response.schema_failed then
            print("[Electrode] Schema failed - AI didn't follow JSON format")
        end
        if response.error then
            print("[Electrode] Response error:", response.error)
        end
    end
    
    if response.assistant_message and response.assistant_message ~= "" then
        appendConversationMessage("assistant", response.assistant_message, { enableTypingIndicator = true })
    elseif response.status then
        appendConversationMessage("assistant", response.status, { enableTypingIndicator = true })
    end

    if response.raw_response then
        local rawPreview = ""
        if typeof(response.raw_response) == "string" then
            rawPreview = response.raw_response
        else
            local ok, encoded = pcall(function()
                return HttpService:JSONEncode(response.raw_response)
            end)
            rawPreview = ok and encoded or tostring(response.raw_response)
        end

        warn("[Electrode] Raw model response:", rawPreview)
        print("[Electrode][RawResponse] " .. rawPreview)

        if rawPreview ~= "" then
            -- Use collapsible format for workspace queries (will be auto-detected by createChatBubble)
            appendConversationMessage("assistant", "Workspace Query Results:\n" .. rawPreview)
        end
    end

    if response.plan then
        -- Validate plan before displaying to filter out error logs
        local validatedPlan = validatePlanEntries(response.plan)
        updatePlanUI(validatedPlan)
    end

    saveAgentSession()
    
    -- Update token usage if response includes token info
    if response.tokens_used and activeTab == "Settings" then
        task.delay(0.5, updateTokenUsage)
    end

    -- Check if this is a chunked response with more batches
    local hasMore = response.has_more == true
    local continueToken = response.continue_token
    local batchInfo = response.batch_info
    
    -- Store continue token for seamless query continuation (even if no more batches)
    if continueToken then
        errorState.currentContinueToken = continueToken
    elseif not hasMore then
        -- Clear token when request fully completes
        errorState.currentContinueToken = nil
    end

    -- Check if we have actions to execute (must exist and have length > 0)
    local hasActions = response.actions and type(response.actions) == "table" and #response.actions > 0
    
    if hasActions then
        -- Check if this response only contains query actions and we've already done a continuation
        -- This prevents infinite query loops
        local hasOnlyQueries = true
        for _, action in ipairs(response.actions) do
            if action.type ~= "query_workspace" then
                hasOnlyQueries = false
                break
            end
        end
        
        -- Handle query-only responses after initial continuation
        if hasOnlyQueries and errorState.queryContinuationDone then
            -- We've already done a query continuation - check if we should allow one more query cycle
            errorState.queryOnlyContinuationCount = (errorState.queryOnlyContinuationCount or 0) + 1
            
            -- Allow up to 1 additional query cycle after the first continuation
            -- This gives the AI a chance to query additional things if needed based on the first query results
            if errorState.queryOnlyContinuationCount > 1 then
                -- Too many query-only continuations - prevent infinite loop
                warn("[Electrode] Too many query-only continuations - stopping to prevent infinite loop")
                appendConversationMessage("assistant", "I've queried the workspace multiple times but didn't generate implementation code. Please review the query results above and try rephrasing your request with more specific details about what you want me to implement.")
                saveAgentSession()
                errorState.queryContinuationPending = false
                errorState.queryContinuationDone = false
                errorState.queryOnlyContinuationCount = 0
                finalizeAgentRequest()
                return
            else
                -- Allow this query cycle - queries will execute and can trigger another continuation
                warn("[Electrode] Continuation returned additional queries (cycle " .. errorState.queryOnlyContinuationCount .. ") - executing queries and allowing one more continuation if needed")
                -- Keep queryContinuationPending = true so queries can trigger continuation
                -- The runAgentActions logic will check queryOnlyContinuationCount to allow one more continuation
            end
        end
        
        -- If this response has implementation actions (not just queries), reset continuation flags
        -- This means we're moving from query phase to implementation phase
        if not hasOnlyQueries then
            errorState.queryContinuationPending = false
            errorState.queryOnlyContinuationCount = 0 -- Reset counter when we get implementation actions
            errorState.queryContinuationDone = false -- Reset so future requests can query if needed
        end
        
        announceActionKickoff(response.actions, batchInfo, isContinuation)
        setThoughtStage(#ThoughtConfig.stages)
        runAgentActions(response.actions, function(success, err)
            -- CRITICAL: Check if query continuation is pending FIRST, before any other logic
            -- This prevents finalization when queries trigger continuation
            if errorState.queryContinuationPending then
                -- Query continuation is processing, don't finalize yet
                -- The continuation request will generate new actions based on query results
                -- Don't show completion message or finalize - just return silently
                -- The continuation will handle finalization when implementation completes
                return
            end
            
            if success then
                if hasMore and continueToken then
                    -- Process next batch automatically
                    local batchNum = batchInfo and batchInfo.batch or 1
                    local totalBatches = batchInfo and batchInfo.total or 1
                    appendConversationMessage("assistant", string.format("Batch %d/%d is in. Grabbing the next chunk now...", batchNum, totalBatches))
                    startLiveNarration(0.8)
                    
                    -- Request next batch with proper staggering (2 seconds between batches)
                    task.delay(2.0, function()
                        local continueOk, continueResponse, continueError = pcall(function()
                            return Network.requestAgentContinue(apiKey, continueToken)
                        end)
                        
                        if continueOk and continueResponse and not continueError then
                            -- Recursively handle the continuation response
                            handleResponse(continueResponse, nil, true)
                        else
                            local errorDescription
                            if not continueOk then
                                errorDescription = Network.formatError(continueResponse)
                            else
                                errorDescription = Network.formatError(continueError or continueResponse)
                            end
                            warn("[Electrode] Continue request failed:", errorDescription)
                            
                            -- Check if it's a token expiration error
                            local isExpired = string.find(errorDescription, "expired", 1, true) or 
                                             string.find(errorDescription, "404", 1, true) or
                                             string.find(errorDescription, "Invalid", 1, true)
                            
                            if isExpired then
                                appendConversationMessage("assistant", "⚠️ The batch token expired. The previous steps completed, but I couldn't continue automatically. You can ask me to continue with the remaining parts.")
                            else
                                appendConversationMessage("assistant", "⚠️ Couldn't grab the next batch: " .. errorDescription .. ". The previous steps completed successfully though.")
                            end
                            
                            saveAgentSession()
                            hideThoughtStages()
                            finalizeAgentRequest()
                        end
                    end)
                else
                    -- All batches complete - clear continue token
                    errorState.currentContinueToken = nil
                    
                    -- Note: queryContinuationPending check already done at top of callback
                    -- If we reach here, continuation is not pending, so safe to finalize
                    
                    -- All actions complete (including implementation) - finalize the request
                    if batchInfo and batchInfo.total and batchInfo.total > 1 then
                        appendConversationMessage("assistant", string.format("That was batch %d/%d - everything's placed in Studio now.", batchInfo.total, batchInfo.total))
                    else
                        appendConversationMessage("assistant", "Dropped the requested changes into Studio.")
                    end
                    saveAgentSession()
                    finalizeAgentRequest()
                end
            elseif err and err ~= "User cancelled" then
                errorState.currentContinueToken = nil -- Clear token on error
                appendConversationMessage("assistant", "Ran into an issue applying the changes: " .. tostring(err))
                saveAgentSession()
                finalizeAgentRequest()
            else
                -- User cancelled - clear token
                errorState.currentContinueToken = nil
                saveAgentSession()
                finalizeAgentRequest()
            end
        end)
    else
        -- No actions - check if this is a query continuation FIRST
        -- This is critical - if continuation is pending, request MUST stay open
        if errorState.queryContinuationPending then
            -- Query continuation is processing, don't finalize yet
            -- The continuation request will generate new actions
            -- Request stays open - don't finalize
            return
        end
        
        -- No actions and not a query continuation - check if schema failed
        local schemaFailed = response.schema_failed == true
        local hasError = response.error ~= nil
        
        -- Check if we have an assistant message that might explain why no actions
        local hasAssistantMessage = response.assistant_message and response.assistant_message ~= ""
        
        if schemaFailed or hasError then
            -- AI didn't generate actions - show helpful message
            local errorMsg = response.error or "The AI didn't generate any actions."
            if schemaFailed then
                errorMsg = "The AI response didn't follow the required JSON format. It may have generated text but not the action JSON needed to execute changes."
            end
            appendConversationMessage("assistant", "⚠️ " .. errorMsg .. " Please try rephrasing your request with more specific details about what you want me to create or modify.")
            warn("[Electrode] No actions in response - schema_failed:", schemaFailed, "error:", response.error)
            if response.raw_response then
                warn("[Electrode] Raw response preview:", tostring(response.raw_response):sub(1, 500))
            end
        elseif hasAssistantMessage then
            -- AI provided a message but no actions - might be asking for clarification
            -- The assistant_message is already displayed above, so just add a helpful note
            appendConversationMessage("assistant", "\n\n💡 **I didn't generate any code changes.** If you want me to implement something, please provide more specific details about:\n\n• What scripts/instances to create\n• What functionality to implement\n• Where things should be placed\n\nFor example: \"Create a script in ServerScriptService that handles player spawning\" is clearer than \"make a spawn system\".")
        else
            -- No actions and no message - completely empty response
            appendConversationMessage("assistant", "I received your request but didn't generate any code changes. This might mean:\n\n• The request was too vague or ambiguous\n• I need more context about what to implement\n• There was an issue processing the request\n\n**Please try rephrasing your request with more specific details.** For example:\n\nInstead of: \"make a gun system\"\nTry: \"Create a pistol model in Workspace with a script that handles shooting, ammo UI in StarterGui, and bullet model for projectiles\"")
        end
        
        -- No actions and not a query continuation - clear continue token if no more batches
        if not hasMore then
            errorState.currentContinueToken = nil
        end
        setThoughtStage(#ThoughtConfig.stages)
        if not hasMore then
            -- Only finalize if continuation is NOT pending
            if not errorState.queryContinuationPending then
                finalizeAgentRequest()
            end
        end
    end
end

function sendAgentMessage()
    if agentRequestInFlight then
        appendConversationMessage("assistant", "Hold up - I'm still finishing the previous run.")
        return
    end

    local rawText = AgentUI.ChatInputBox.Text or ""
    local trimmed = rawText:gsub("^%s+", ""):gsub("%s+$", "")
    if trimmed == "" then
        return
    end

    local apiKey = getActiveApiKey()
    if apiKey == "" then
        appendConversationMessage("assistant", "I'll need your API key before we keep going. Please enter it in the Settings tab.")
        showApiKeyPopup()
        saveAgentSession()
        return
    end

    -- Reset query continuation flags when starting a new user message
    errorState.queryContinuationDone = false
    errorState.queryContinuationPending = false
    errorState.queryOnlyContinuationCount = 0

    AgentUI.ChatInputBox.Text = ""
    appendConversationMessage("user", trimmed)
    saveAgentSession()

    agentRequestInFlight = true
    agentState.pendingAssistantChunks = 0
    AgentUI.SendButton.Text = "Thinking..."
    AgentUI.SendButton.BackgroundColor3 = Color3.fromRGB(108, 117, 125)
    AgentUI.SendButton.Active = false
    startLiveNarration()
    showThoughtStages()
    setThoughtStage(1)

    local function performRequest(snapshot)
        print("[Electrode] Parsing prompt and preparing request for backend...")
        print("[Electrode] User prompt:", trimmed)
        
        local payload = {
            api_key = apiKey,
            model = agentState.model,
            trusted = agentState.trusted,
            session_id = agentState.sessionId,
            conversation = serializeConversation(),
            snapshot = buildSnapshotPayload(snapshot),
            plan = agentState.plan,
            roblox_user_id = Network.getRobloxUserId(),
            request = {
                message = trimmed,
                timestamp = os.time()
            }
        }

        print("[Electrode] Payload prepared, sending to backend...")
        print("[Electrode] Backend will retrieve relevant context blocks based on prompt tags")
        
        local ok, response, httpError = pcall(function()
            setThoughtStage(3)
            print("[Electrode] Request sent to backend endpoint: /api/agent/chat")
            return Network.requestAgentResponse(apiKey, payload)
        end)

        if not ok then
            handleResponse(nil, response)
        else
            handleResponse(response, httpError)
        end
    end

    local function proceedWithSnapshot(snapshot)
        setThoughtStage(3)
        performRequest(snapshot or snapshotState.current)
    end

    if snapshotState.capturing then
        setThoughtStage(2)
        captureSnapshot("agent-request", proceedWithSnapshot)
    elseif snapshotState.current and not snapshotState.dirty then
        setThoughtStage(2)
        proceedWithSnapshot(snapshotState.current)
    else
        setThoughtStage(2)
        captureSnapshot("agent-request", proceedWithSnapshot)
    end
end

local function updateDiffUI(diff)
    clearScrollingFrame(AgentUI.DiffScroll)

    if not diff or (#(diff.added or {}) == 0 and #(diff.removed or {}) == 0 and #(diff.modified or {}) == 0) then
        local label = Instance.new("TextLabel")
        label.BackgroundTransparency = 1
        label.Size = UDim2.new(1, 0, 0, 24)
        label.TextXAlignment = Enum.TextXAlignment.Left
        label.Font = Enum.Font.SourceSans
        label.TextSize = 13
        label.TextColor3 = Color3.fromRGB(104, 113, 122)
        label.Text = "No recent changes detected."
        label.Parent = AgentUI.DiffScroll
        snapshotState.diff = { added = {}, removed = {}, modified = {} }
        task.defer(updateDiffCanvasSize)
        return
    end

    local function addDiffLabel(prefix, entry, color)
        local label = Instance.new("TextLabel")
        label.BackgroundTransparency = 1
        label.Size = UDim2.new(1, 0, 0, 24)
        label.TextXAlignment = Enum.TextXAlignment.Left
        label.Font = Enum.Font.SourceSans
        label.TextSize = 13
        label.TextColor3 = color
        label.Text = string.format("%s %s (%s)", prefix, entry.path or entry.name or "?", entry.class or entry.kind or "")
        label.Parent = AgentUI.DiffScroll
    end

    for _, entry in ipairs(diff.added or {}) do
        addDiffLabel("+ Added", entry, STATUS_COLORS.success)
    end

    for _, entry in ipairs(diff.removed or {}) do
        addDiffLabel("− Removed", entry, Color3.fromRGB(220, 53, 69))
    end

    for _, entry in ipairs(diff.modified or {}) do
        addDiffLabel("~ Updated", entry, STATUS_COLORS.warning)
    end

    snapshotState.diff = diff
    task.defer(updateDiffCanvasSize)
end

local function updateSnapshotSummary(snapshot)
    if not snapshot then
        AgentUI.SnapshotTimestampLabel.Text = "Snapshot: not captured"
        AgentUI.SnapshotSummaryLabel.Text = "No recent hierarchy summary."
        return
    end

    AgentUI.SnapshotTimestampLabel.Text = string.format(
        "Snapshot: %s",
        os.date("%Y-%m-%d %H:%M:%S", snapshot.timestamp or os.time())
    )

    local summaryParts = {}
    if snapshot.summary then
        -- Safely convert all values to numbers/strings to prevent type errors
        if snapshot.summary.totalInstances then
            local count = tonumber(snapshot.summary.totalInstances) or 0
            table.insert(summaryParts, string.format("%d objects", count))
        end
        if snapshot.summary.scriptCount then
            local count = tonumber(snapshot.summary.scriptCount) or 0
            table.insert(summaryParts, string.format("%d scripts", count))
        end
        if snapshot.summary.remoteCount then
            local count = tonumber(snapshot.summary.remoteCount) or 0
            table.insert(summaryParts, string.format("%d remotes", count))
        end
        if snapshot.summary.services then
            for _, entry in ipairs(snapshot.summary.services) do
                -- Ensure name and count are valid (not nil, not boolean)
                local name = tostring(entry.name or "Unknown")
                local count = tonumber(entry.count) or 0
                table.insert(summaryParts, string.format("%s: %d", name, count))
            end
        end
    end

    if #summaryParts == 0 then
        AgentUI.SnapshotSummaryLabel.Text = "Snapshot captured."
    else
        AgentUI.SnapshotSummaryLabel.Text = table.concat(summaryParts, " • ")
    end
end

local importantServices = {
    "ServerScriptService",
    "ReplicatedStorage",
    "StarterPlayer",
    "StarterGui",
    "ServerStorage",
    "Workspace"
}

local function collectHierarchySnapshot(maxNodes)
    maxNodes = maxNodes or 600
    local queue = {}
    local flatMap = {}
    local processed = 0
    
    -- Collect from all important services, not just Workspace
    local serviceRoots = {}
    
    -- Add Workspace
    table.insert(serviceRoots, workspace)
    flatMap["Workspace"] = "Workspace"
    
    -- Add other important services
    for _, serviceName in ipairs(importantServices) do
        if serviceName ~= "Workspace" then
            local success, service = pcall(function()
                return game:GetService(serviceName)
            end)
            if success and service then
                table.insert(serviceRoots, service)
                flatMap[serviceName] = serviceName
            end
        end
    end
    
    -- Start queue with all service roots
    for _, root in ipairs(serviceRoots) do
        table.insert(queue, root)
    end

    while #queue > 0 and processed < maxNodes do
        local current = table.remove(queue, 1)
        processed += 1

        for _, child in ipairs(current:GetChildren()) do
            local path = child:GetFullName()

            if not child:IsA("LuaSourceContainer") then
                flatMap[path] = child.ClassName
            end

            table.insert(queue, child)
        end
    end

    return flatMap
end

local function buildSnapshotSummary(scriptCount, remotes)
    local servicesSummary = {}

    for _, serviceName in ipairs(importantServices) do
        local ok, service = pcall(function()
            return game:GetService(serviceName)
        end)

        if ok and service then
            table.insert(servicesSummary, {
                name = serviceName,
                count = #service:GetDescendants()
            })
        end
    end

    local totalInstances = #game:GetDescendants() + 1

    return {
        totalInstances = totalInstances,
        scriptCount = scriptCount,
        remoteCount = remotes and #remotes or 0,
        services = servicesSummary
    }
end

local function computeSnapshotDiff(newSnapshot, oldSnapshot)
    if not newSnapshot then
        return { added = {}, removed = {}, modified = {} }
    end

    if not oldSnapshot then
        local added = {}
        for path, className in pairs(newSnapshot.map or {}) do
            if path ~= "Workspace" then
                table.insert(added, { path = path, class = className })
            end
        end
        for path, info in pairs(newSnapshot.scriptsMap or {}) do
            table.insert(added, { path = path, class = info.class, kind = "script" })
        end
        return { added = added, removed = {}, modified = {} }
    end

    local diff = {
        added = {},
        removed = {},
        modified = {}
    }

    for path, className in pairs(newSnapshot.map or {}) do
        local previous = oldSnapshot.map and oldSnapshot.map[path]
        if not previous then
            table.insert(diff.added, { path = path, class = className })
        elseif previous ~= className then
            table.insert(diff.modified, { path = path, class = className })
        end
    end

    for path, className in pairs(oldSnapshot.map or {}) do
        if not (newSnapshot.map and newSnapshot.map[path]) then
            table.insert(diff.removed, { path = path, class = className })
        end
    end

    for path, info in pairs(newSnapshot.scriptsMap or {}) do
        local previous = oldSnapshot.scriptsMap and oldSnapshot.scriptsMap[path]
        if not previous then
            table.insert(diff.added, { path = path, class = info.class, kind = "script" })
        elseif previous.hash ~= info.hash then
            table.insert(diff.modified, { path = path, class = info.class, kind = "script" })
        end
    end

    for path, info in pairs(oldSnapshot.scriptsMap or {}) do
        if not (newSnapshot.scriptsMap and newSnapshot.scriptsMap[path]) then
            table.insert(diff.removed, { path = path, class = info.class, kind = "script" })
        end
    end

    return diff
end

-- Attach to snapshotState to avoid module-level locals
snapshotState.cloneCallbacks = function(self, list)
    local copy = {}
    for index = 1, #list do
        copy[index] = list[index]
    end
    return copy
end

snapshotState.dispatchCallbacks = function(self, snapshot, diff, err)
    if #self.callbacks == 0 then
        return
    end

    local callbacks = self:cloneCallbacks(self.callbacks)
    table.clear(self.callbacks)

    for _, cb in ipairs(callbacks) do
        local okCallback, callbackErr = pcall(cb, snapshot, diff, err)
        if not okCallback then
            warn("Snapshot callback error:", callbackErr)
        end
    end
end

captureSnapshot = function(reason, callback)
    reason = reason or "manual"

    if callback then
        table.insert(snapshotState.callbacks, callback)
    end

    if snapshotState.capturing then
        return
    end

    snapshotState.capturing = true
    snapshotState.lastReason = reason

    task.spawn(function()
        local ok, err = pcall(function()
            local success, scriptCount = pcall(cacheAllScripts)
            if not success then
                scriptCount = 0
            end

            local remotes = {}
            local remoteOk, remoteResult = pcall(scriptAnalysis.findAllRemotes)
            if remoteOk and type(remoteResult) == "table" then
                remotes = remoteResult
            end

            local map = collectHierarchySnapshot()
            local scriptsMap = {}
            for _, scriptInfo in ipairs(scriptCache) do
                scriptsMap[scriptInfo.path] = {
                    class = scriptInfo.className or "Script",
                    hash = scriptInfo.sourceHash,
                    name = scriptInfo.name
                }
            end

            local snapshot = {
                timestamp = os.time(),
                summary = buildSnapshotSummary(scriptCount, remotes),
                map = map,
                scriptsMap = scriptsMap,
                remotes = remotes,
                reason = reason
            }

            snapshotState.previous = snapshotState.current
            snapshotState.current = snapshot
            snapshotState.dirty = false
            snapshotState.capturing = false

            local diff = computeSnapshotDiff(snapshotState.current, snapshotState.previous)
            updateSnapshotSummary(snapshotState.current)
            updateDiffUI(diff)
            agentState.diffEntries = diff

            saveAgentSession()

            snapshotState:dispatchCallbacks(snapshotState.current, diff, nil)
        end)

        if not ok then
            snapshotState.capturing = false
            snapshotState.dirty = true
            warn("Snapshot capture failed:", err)
            snapshotState:dispatchCallbacks(nil, nil, err)
        end
    end)
end

-- Attach to snapshotState to avoid module-level local
snapshotState.schedule = function(self, reason)
    self.dirty = true
    self.lastReason = reason or self.lastReason

    if self.scheduled then
        return
    end

    self.scheduled = true
    task.delay(1.5, function()
        self.scheduled = false
        if self.dirty and not self.capturing then
            captureSnapshot(reason or "auto")
        end
    end)
end
local function scheduleSnapshot(reason)
    return snapshotState:schedule(reason)
end

local function shouldTrackInstance(instance)
    if instance:IsA("LuaSourceContainer") then
        return true
    end
    if instance:IsA("RemoteEvent") or instance:IsA("RemoteFunction") then
        return true
    end
    local parent = instance.Parent
    if parent == workspace or parent == nil then
        return true
    end
    return false
end

game.DescendantAdded:Connect(function(instance)
    if shouldTrackInstance(instance) then
        scheduleSnapshot("descendant-added")
    end
end)

game.DescendantRemoving:Connect(function(instance)
    if shouldTrackInstance(instance) then
        scheduleSnapshot("descendant-removed")
    end
end)

function buildSnapshotPayload(snapshot)
    if not snapshot then
        return nil
    end

    local hierarchyEntries = {}
    if snapshot.map then
        local count = 0
        for path, className in pairs(snapshot.map) do
            table.insert(hierarchyEntries, { path = path, class = className })
            count += 1
            if count >= 400 then
                break
            end
        end
        table.sort(hierarchyEntries, function(a, b)
            return a.path < b.path
        end)
    end

    local scriptEntries = {}
    -- Limit to top 100 scripts to prevent timeout (use query_workspace for full access)
    -- AI can use query_workspace to read any script it needs with full source
    local scriptCache = scriptAnalysis.scriptCache or {}
    local maxScripts = math.min(100, #scriptCache)
    for i = 1, maxScripts do
        local scriptInfo = scriptCache[i]
        if scriptInfo then
            table.insert(scriptEntries, {
                path = scriptInfo.path,
                class = scriptInfo.className,
                hash = scriptInfo.sourceHash,
                lineCount = scriptInfo.lineCount,
                snippet = scriptInfo.source:sub(1, 800) -- Still snippet for snapshot, but query_workspace returns full source
            })
        end
    end

    -- METHOD 4: Enhanced Snapshot Payload - Include script analysis and dependency graph
    -- Limit to top 100 scripts to prevent timeout (use query_workspace for full access)
    local scriptAnalysisData = {}
    for i = 1, maxScripts do
        local scriptInfo = scriptCache[i]
        if scriptInfo and scriptInfo.analysis then
            scriptAnalysisData[scriptInfo.path] = {
                functions = scriptInfo.analysis.functions,
                exports = scriptInfo.analysis.exports,
                requires = scriptInfo.analysis.requires,
                services = scriptInfo.analysis.services,
                patterns = scriptInfo.analysis.patterns
            }
        end
    end

    -- Build dependency graph (use module-level scriptAnalysis table)
    -- Only build graph for scripts we're including in snapshot to reduce size
    local dependencyGraph = {}
    local success, fullDependencyGraph = pcall(function()
        return scriptAnalysis.buildDependencyGraph()
    end)
    
    if success and fullDependencyGraph and type(fullDependencyGraph) == "table" then
        -- Filter dependency graph to only include scripts in our snapshot
        for path, depsInfo in pairs(fullDependencyGraph) do
            -- Include if this script is in our snapshot
            local includeScript = false
            if maxScripts > 0 and scriptCache then
                for i = 1, maxScripts do
                    local scriptInfo = scriptCache[i]
                    if scriptInfo and scriptInfo.path == path then
                        includeScript = true
                        break
                    end
                end
            end
            if includeScript then
                dependencyGraph[path] = depsInfo
            end
        end
    end

    return {
        summary = snapshot.summary,
        remotes = snapshot.remotes,
        hierarchy = hierarchyEntries,
        scripts = scriptEntries,
        scriptAnalysis = scriptAnalysisData, -- NEW: Deep analysis of all scripts
        dependencyGraph = dependencyGraph, -- NEW: Dependency relationships
        diff = snapshotState.diff
    }
end

function serializeConversation()
    local serialized = {}
    for _, message in ipairs(agentState.conversation) do
        table.insert(serialized, {
            role = message.role,
            content = message.content,
            timestamp = message.timestamp,
            notes = message.notes
        })
    end
    return serialized
end

function saveAgentSession()
    -- Check if auto-save is enabled
    local autoSaveEnabled = true -- Default to enabled
    if SettingsUI.AutoSaveToggle then
        autoSaveEnabled = SettingsUI.AutoSaveToggle:GetAttribute("Enabled") == true
    end
    
    if not autoSaveEnabled then
        return -- Don't save if auto-save is disabled
    end
    
    local payload = {
        conversation = serializeConversation(),
        plan = agentState.plan,
        trusted = agentState.trusted,
        model = agentState.model,
        sessionId = agentState.sessionId,
        diff = snapshotState.diff,
        snapshotTimestamp = snapshotState.current and snapshotState.current.timestamp or nil,
        snapshotSummaryText = AgentUI.SnapshotSummaryLabel.Text,
        snapshotLabel = AgentUI.SnapshotTimestampLabel.Text
    }

    pcall(function()
        plugin:SetSetting(AGENT_SESSION_KEY, payload)
    end)
end

function loadAgentSession()
    local saved = nil
    local ok, result = pcall(function()
        return plugin:GetSetting(AGENT_SESSION_KEY)
    end)

    if ok and type(result) == "table" then
        saved = result
    end

    if saved then
        agentState.conversation = saved.conversation or agentState.conversation
        -- Validate plan when loading from saved session
        local savedPlan = saved.plan
        if savedPlan and type(savedPlan) == "table" then
            agentState.plan = validatePlanEntries(savedPlan) or {}
        else
            agentState.plan = {}
        end
        agentState.trusted = saved.trusted == true
        if saved.model then
            agentState.model = saved.model
        end
        if saved.sessionId then
            agentState.sessionId = saved.sessionId
        end
        reloadChatFromState()
        updatePlanUI(agentState.plan)
        updateDiffUI(saved.diff)
        if saved.snapshotTimestamp or saved.snapshotSummaryText then
            -- Ensure saved values are strings, not booleans or other types
            local savedLabel = saved.snapshotLabel
            local savedSummary = saved.snapshotSummaryText
            if savedLabel and typeof(savedLabel) == "string" then
                AgentUI.SnapshotTimestampLabel.Text = savedLabel
            end
            if savedSummary and typeof(savedSummary) == "string" then
                AgentUI.SnapshotSummaryLabel.Text = savedSummary
            end
        end
    else
        updateDiffUI(nil)
        updatePlanUI({})
    end

    setAgentModel(agentState.model, true)
    setTrustedMode(agentState.trusted)
    task.defer(updateChatCanvasSize)
end

function getActiveApiKey()
    if cachedApiKey and cachedApiKey ~= "" then
        return cachedApiKey
    end

    -- Settings API key input removed - using popup only

    local success, saved = pcall(function()
        return plugin:GetSetting("ApiKey")
    end)

    if success and saved and saved ~= "" then
        cachedApiKey = saved
        -- Settings API key input removed - using popup only
        return saved
    end

    return ""
end
local placementStopWords = {
    ["the"] = true, ["and"] = true, ["for"] = true, ["that"] = true, ["with"] = true,
    ["your"] = true, ["this"] = true, ["make"] = true, ["have"] = true, ["will"] = true,
    ["script"] = true, ["code"] = true, ["please"] = true, ["from"] = true, ["into"] = true,
    ["when"] = true, ["then"] = true, ["also"] = true, ["like"] = true, ["just"] = true,
    ["need"] = true, ["want"] = true, ["auto"] = true, ["placement"] = true,
    ["module"] = true, ["modulescript"] = true, ["localscript"] = true, ["serverscript"] = true,
    ["called"] = true, ["named"] = true, ["name"] = true, ["folder"] = true, ["service"] = true,
    ["create"] = true, ["creates"] = true, ["creating"] = true,
    ["generate"] = true, ["generates"] = true, ["generating"] = true,
    ["let"] = true, ["lets"] = true, ["allow"] = true, ["allows"] = true, ["allowing"] = true,
    ["user"] = true, ["users"] = true, ["player"] = true, ["players"] = true,
    ["press"] = true, ["presses"] = true, ["pressing"] = true,
    ["hold"] = true, ["holding"] = true, ["start"] = true, ["starting"] = true,
    ["cause"] = true, ["causes"] = true, ["causing"] = true,
    ["toggle"] = true, ["toggling"] = true, ["enable"] = true, ["enables"] = true, ["enabling"] = true,
    ["disable"] = true, ["disables"] = true, ["disabling"] = true,
    ["give"] = true, ["gives"] = true, ["giving"] = true,
    ["makes"] = true, ["making"] = true
}

local PLACEMENT_DEFAULT_MESSAGE = "Electrode will suggest where to place this script."

local function stripComments(text)
    if not text or text == "" then
        return ""
    end

    local withoutBlock = text:gsub("%-%-%[%[.-%]%]", "")
    local withoutLine = withoutBlock:gsub("%-%-.-\n", "\n")
    return withoutLine
end

-- Placement button functions removed (Generator functionality)

local function sanitizePromptWords(prompt)
    local words = {}
    if not prompt or prompt == "" then
        return words
    end

    for word in prompt:lower():gmatch("%w+") do
        if #word > 2 and not placementStopWords[word] then
            table.insert(words, word)
        end
    end

    return words
end

local function normalizeNameWord(word)
    if not word or word == "" then
        return nil
    end

    local cleaned = word:gsub("[^%w]", "")
    if cleaned == "" then
        return nil
    end

    if cleaned:match("ing$") and #cleaned > 5 then
        cleaned = cleaned:sub(1, #cleaned - 3)
    elseif cleaned:match("ers$") and #cleaned > 4 then
        cleaned = cleaned:sub(1, #cleaned - 1)
    elseif cleaned:match("ies$") and #cleaned > 4 then
        cleaned = cleaned:sub(1, #cleaned - 3) .. "y"
    end

    if cleaned == "" then
        return nil
    end

    cleaned = cleaned:sub(1, 1):upper() .. cleaned:sub(2)
    return cleaned
end

local function applyScriptSuffix(name, scriptType, skipSuffix)
    if not name or name == "" then
        return name
    end

    if skipSuffix then
        return name
    end

    local lower = name:lower()
    if scriptType == "ModuleScript" then
        if not lower:find("module", 1, true) then
            return name .. "Module"
        end
    else
        if not lower:find("script", 1, true) then
            return name .. "Script"
        end
    end

    return name
end

local function generateScriptNameFromPrompt(prompt, scriptType)
    local directName = nil

    if prompt and prompt ~= "" then
        directName = prompt:match("[cC]alled%s+([%w_]+)")
            or prompt:match("[nN]amed%s+([%w_]+)")
            or prompt:match("`([%w_]+)`")
            or prompt:match('"([%w_]+)"')
    end

    if directName and directName ~= "" then
        local title = directName:gsub("(%a)(%w*)", function(first, rest)
            return first:upper() .. rest:lower()
        end)
        if #title > 32 then
            title = title:sub(1, 32)
        end

        return applyScriptSuffix(title, scriptType, false)
    end

    local promptLower = ""
    if prompt and prompt ~= "" then
        promptLower = prompt:lower()
    end

    if promptLower ~= "" then
        local hasShift = promptLower:find("shift", 1, true)
        local hasRun = promptLower:find("run", 1, true)
        local hasSprint = promptLower:find("sprint", 1, true)

        if hasShift and hasSprint then
            return applyScriptSuffix("ShiftSprint", scriptType, false)
        elseif hasShift and hasRun then
            return applyScriptSuffix("ShiftToRun", scriptType, false)
        elseif hasSprint and not hasShift then
            return applyScriptSuffix("Sprint", scriptType, false)
        end
    end

    local words = sanitizePromptWords(prompt)
    local nameParts = {}
    local seen = {}
    local maxParts = 2

    for _, word in ipairs(words) do
        if not seen[word] then
            local title = normalizeNameWord(word)
            if title then
                table.insert(nameParts, title)
            end
            seen[word] = true
        end

        if #nameParts >= maxParts then
            break
        end
    end

    if #nameParts == 0 then
        table.insert(nameParts, scriptType == "ModuleScript" and "Utility" or "Generated")
    end

    local rawName = table.concat(nameParts)

    if rawName == "" then
        rawName = scriptType == "ModuleScript" and "ElectrodeModule" or "ElectrodeScript"
    end

    rawName = rawName:gsub("%W", "")

    if #rawName > 32 then
        rawName = rawName:sub(1, 32)
    end

    local finalName = applyScriptSuffix(rawName, scriptType, false)
    if finalName and finalName ~= "" then
        return finalName
    end

    if scriptType == "ModuleScript" then
        return "ElectrodeModule"
    end

    return "ElectrodeScript"
end

local function detectScriptTypeFromCode(code)
    if not code or code == "" then
        return "Script"
    end

    local sanitized = stripComments(code)
    local lower = sanitized:lower()

    local lastLine = ""
    for line in sanitized:gmatch("[^\n]+") do
        local trimmed = line:match("^%s*(.-)%s*$")
        if trimmed ~= "" then
            lastLine = trimmed
        end
    end

    if lastLine ~= "" then
        local lastLower = lastLine:lower()
        if lastLower:match("^return%s+[%w_]+") or lastLower:match("^return%s+{") then
            return "ModuleScript"
        end
    end

    if lower:find("module%s*=") or lower:find("return%s+module") or lower:find("return%s+{") or lower:find("return%s+function") then
        return "ModuleScript"
    end

    if lower:find("players%.localplayer") or lower:find("userinputservice") or lower:find("contextactionservice")
        or lower:find("playergui") or lower:find("startergui") or lower:find("runservice%.renderstepped") then
        return "LocalScript"
    end

    return "Script"
end

local function suggestTargetPath(scriptType, codeLower)
    if scriptType == "LocalScript" then
        if codeLower:find("playergui") or codeLower:find("startergui") or codeLower:find("screen%p?gui") or codeLower:find("textbutton") then
            return {"StarterGui"}
        end

        if codeLower:find("startercharacterscripts") or (codeLower:find("character") and codeLower:find("humanoid")) then
            return {"StarterPlayer", "StarterCharacterScripts"}
        end

        return {"StarterPlayer", "StarterPlayerScripts"}
    elseif scriptType == "ModuleScript" then
        if codeLower:find("serverscriptservice") and not codeLower:find("replicatedstorage") then
            return {"ServerScriptService"}
        end
        return {"ReplicatedStorage"}
    else -- Script
        if codeLower:find("workspace") or codeLower:find("serverscriptservice") or codeLower:find("datastoreservice") or codeLower:find("collectionservice") then
            return {"ServerScriptService"}
        end

        if codeLower:find("replicatedstorage") then
            return {"ReplicatedStorage"}
        end

        return {"ServerScriptService"}
    end
end

local function validateWorkspacePath(segments)
    if not segments or #segments == 0 then
        return nil, nil
    end

    local current = workspace
    local lastInstance = workspace
    local pathParts = {"Workspace"}

    for index, segment in ipairs(segments) do
        if not current then
            return nil, nil
        end

        local child = current:FindFirstChild(segment)

        if child then
            current = child
            lastInstance = child
            table.insert(pathParts, segment)
        else
            if index == 1 then
                return nil, nil
            else
                return pathParts, lastInstance
            end
        end
    end

    return pathParts, lastInstance
end

local function findWorkspacePlacementTarget(code)
    if not code or code == "" then
        return nil, nil
    end

    local references = {}

    for childName in code:gmatch("[Ww]orkspace%s*[:%.]%s*WaitForChild%s*%(%s*[\"']([%w_]+)[\"']%s*%)") do
        table.insert(references, {childName})
    end

    for childName in code:gmatch("[Ww]orkspace%s*[:%.]%s*FindFirstChild%s*%(%s*[\"']([%w_]+)[\"']%s*%)") do
        table.insert(references, {childName})
    end

    for path in code:gmatch("[Ww]orkspace%.([%w_%.]+)") do
        local segments = {}
        for segment in path:gmatch("([%w_]+)") do
            table.insert(segments, segment)
        end
        if #segments > 0 then
            table.insert(references, segments)
        end
    end

    for path in code:gmatch("[Gg]ame%.Workspace%.([%w_%.]+)") do
        local segments = {}
        for segment in path:gmatch("([%w_]+)") do
            table.insert(segments, segment)
        end
        if #segments > 0 then
            table.insert(references, segments)
        end
    end

    for chain in code:gmatch("[Ww]orkspace%s*%[[^%]]+%][%w_%.%[%]'\"]*") do
        local segments = {}
        for segment in chain:gmatch("%[[\"']([%w_]+)[\"']%]") do
            table.insert(segments, segment)
        end
        if chain:match("%.([%w_]+)") then
            for dotSegment in chain:gmatch("%.([%w_]+)") do
                table.insert(segments, dotSegment)
            end
        end
        if #segments > 0 then
            table.insert(references, segments)
        end
    end

    for chain in code:gmatch("[Gg]ame%.Workspace%s*%[[^%]]+%][%w_%.%[%]'\"]*") do
        local segments = {}
        for segment in chain:gmatch("%[[\"']([%w_]+)[\"']%]") do
            table.insert(segments, segment)
        end
        if chain:match("%.([%w_]+)") then
            for dotSegment in chain:gmatch("%.([%w_]+)") do
                table.insert(segments, dotSegment)
            end
        end
        if #segments > 0 then
            table.insert(references, segments)
        end
    end

    for _, segments in ipairs(references) do
        local pathParts, instance = validateWorkspacePath(segments)
        if pathParts and instance and instance ~= workspace then
            if instance:IsA("BasePart") or instance:IsA("Folder") or instance:IsA("Model") or instance:IsA("Tool") or instance:IsA("Accessory") then
                return pathParts, instance
            end
        end
    end

    return nil, nil
end

local function getPathSegmentsFromInstance(instance)
    if not instance then
        return nil
    end

    local fullName = instance:GetFullName()
    local segments = {}

    for segment in string.gmatch(fullName, "[^%.]+") do
        table.insert(segments, segment)
    end

    return segments
end

local function findUniqueInstancePathByName(name)
    if not name or name == "" then
        return nil, nil
    end

    local foundInstance = nil
    local count = 0

    for _, obj in ipairs(game:GetDescendants()) do
        if obj.Name == name and not obj:IsA("LuaSourceContainer") then
            count += 1
            if count > 1 then
                return nil, nil
            end
            foundInstance = obj
        end
    end

    if foundInstance and count == 1 then
        local segments = getPathSegmentsFromInstance(foundInstance)
        return segments, foundInstance
    end

    return nil, nil
end

local function extractCandidateInstanceNames(prompt, code)
    local candidates = {}
    local function addCandidate(value)
        if not value or value == "" then
            return
        end
        if #value < 3 then
            return
        end
        candidates[value] = true
    end

    if code and code ~= "" then
        for name in code:gmatch("[Ww]aitForChild%s*%(%s*['\"]([%w_]+)['\"]%s*%)") do
            addCandidate(name)
        end

        for name in code:gmatch("[Ff]indFirstChild%s*%(%s*['\"]([%w_]+)['\"]%s*%)") do
            addCandidate(name)
        end
    end

    if prompt and prompt ~= "" then
        for name in prompt:gmatch("`([%w_]+)`") do
            addCandidate(name)
        end

        for name in prompt:gmatch('"([%w_]+)"') do
            addCandidate(name)
        end

        for word in prompt:gmatch("([%w_]+)") do
            if word:match("%u") and not placementStopWords[word:lower()] then
                addCandidate(word)
            end
        end
    end

    local list = {}
    for name in pairs(candidates) do
        table.insert(list, name)
    end

    table.sort(list, function(a, b)
        return #a > #b
    end)

    return list
end

local function resolveTargetPath(pathParts)
    if not pathParts or #pathParts == 0 then
        return nil
    end

    local current
    for index, part in ipairs(pathParts) do
        if index == 1 then
            local success, service = pcall(function()
                return game:GetService(part)
            end)

            if success and service then
                current = service
            else
                return nil
            end
        else
            if current then
                current = current:FindFirstChild(part)
            end

            if not current then
                return nil
            end
        end
    end

    return current
end

local function splitPathString(pathString)
    local segments = {}
    if not pathString or pathString == "" then
        return segments
    end

    for segment in string.gmatch(pathString, "[^%.]+") do
        table.insert(segments, segment)
    end

    if segments[1] and segments[1]:lower() == "game" then
        table.remove(segments, 1)
    end

    return segments
end

local function resolveInstanceFromSegments(segments, options)
    options = options or {}
    if #segments == 0 then
        return nil, "No path provided"
    end

    local current = game
    for index, segment in ipairs(segments) do
        local isLast = index == #segments

        if current == game then
            -- Handle special containers that aren't services
            local segmentLower = segment:lower()
            if segmentLower == "workspace" then
                current = workspace
            elseif segmentLower == "starterplayer" then
                current = game:FindFirstChild("StarterPlayer")
                if not current then
                    -- Try alternative: StarterPlayer might be accessed differently
                    local success, alt = pcall(function()
                        return game.StarterPlayer
                    end)
                    if success and alt then
                        current = alt
                    else
                        return nil, "StarterPlayer not found. It may not exist in this place."
                    end
                end
            elseif segmentLower == "startergui" then
                current = game:FindFirstChild("StarterGui")
                if not current then
                    local success, alt = pcall(function()
                        return game.StarterGui
                    end)
                    if success and alt then
                        current = alt
                    else
                        return nil, "StarterGui not found. It may not exist in this place."
                    end
                end
            elseif segmentLower == "starterpack" then
                current = game:FindFirstChild("StarterPack")
                if not current then
                    local success, alt = pcall(function()
                        return game.StarterPack
                    end)
                    if success and alt then
                        current = alt
                    else
                        return nil, "StarterPack not found. It may not exist in this place."
                    end
                end
            else
                -- Try as a service first
                local success, service = pcall(function()
                    return game:GetService(segment)
                end)
                if success and service then
                    current = service
                else
                    -- Fallback: try as a direct child of game
                    local directChild = game:FindFirstChild(segment)
                    if directChild then
                        current = directChild
                    else
                        return nil, "Unknown service or container: " .. segment .. ". Available: Check game services or containers."
                    end
                end
            end
        else
            local child = current:FindFirstChild(segment)
            if not child and options.createMissing and (not isLast or options.createLeaf) then
                local className = options.intermediateClass or "Folder"
                if isLast and options.leafClass then
                    className = options.leafClass
                end

                local success, newChild = pcall(function()
                    local instance = Instance.new(className)
                    instance.Name = segment
                    instance.Parent = current
                    return instance
                end)

                if success and newChild then
                    child = newChild
                    -- Verify the child is immediately accessible via FindFirstChild
                    -- This ensures subsequent actions can find the created instance
                    local verifyChild = current:FindFirstChild(segment)
                    if verifyChild ~= child then
                        warn("Warning: Created " .. className .. " '" .. segment .. "' not immediately accessible at " .. current:GetFullName())
                        -- Still use the created instance, but log the warning
                    end
                else
                    return nil, "Failed to create " .. className .. " at " .. current:GetFullName()
                end
            end

            if not child then
                return nil, "Missing instance: " .. segment .. " under " .. current:GetFullName()
            end

            current = child
        end
    end

    return current
end

local function resolveInstanceFromPath(pathString, options)
    local segments = splitPathString(pathString)
    return resolveInstanceFromSegments(segments, options)
end

local function determinePlacementContext(prompt, code)
    local context = {
        scriptType = "Script",
        scriptClass = "Script",
        suggestedName = "ElectrodeScript",
        targetPath = nil,
        targetDisplay = "—",
        requiresManual = true,
        reason = "Placement not available.",
        autoPlaceEnabled = false
    }

    if not code or code == "" then
        context.reason = "No code generated."
        return context
    end

    local scriptType = detectScriptTypeFromCode(code)
    context.scriptType = scriptType
    context.scriptClass = scriptType

    context.suggestedName = generateScriptNameFromPrompt(prompt, scriptType)

    local sanitizedCode = stripComments(code)
    local codeLower = sanitizedCode:lower()
    local workspaceTargetPath = nil
    local foundWorkspacePath = select(1, findWorkspacePlacementTarget(code))

    if foundWorkspacePath and scriptType == "Script" then
        workspaceTargetPath = foundWorkspacePath
    end

    local uniqueTargetPath = nil
    local candidateNames = extractCandidateInstanceNames(prompt, code)

    for _, candidate in ipairs(candidateNames) do
        local pathSegments, instance = findUniqueInstancePathByName(candidate)
        if pathSegments and instance then
            if scriptType == "LocalScript" then
                if instance:IsA("GuiBase2d") or instance:IsA("GuiObject") or instance:IsA("LayerCollector") or instance:IsA("BasePart") or instance:IsA("Model") or instance:IsA("Folder") or instance:IsA("Tool") then
                    uniqueTargetPath = pathSegments
                    break
                end
            else
                if not instance:IsA("LuaSourceContainer") then
                    uniqueTargetPath = pathSegments
                    break
                end
            end
        end
    end

    local explicitTargetPath = workspaceTargetPath or uniqueTargetPath
    local targetPath = explicitTargetPath or suggestTargetPath(scriptType, codeLower)
    context.targetPath = targetPath
    context.targetDisplay = targetPath and table.concat(targetPath, " → ") or "—"

    local usesScriptParent = codeLower:find("script%.parent") ~= nil
    local allowScriptParentAuto = false

    if usesScriptParent and targetPath then
        local first = targetPath[1]
        local second = targetPath[2]

        if first == "StarterPlayer" then
            if second == "StarterPlayerScripts" or second == "StarterCharacterScripts" then
                allowScriptParentAuto = true
            end
        elseif first == "StarterGui" then
            allowScriptParentAuto = true
        end
    end

    if explicitTargetPath then
        local leafName = explicitTargetPath[#explicitTargetPath]
        if leafName and leafName ~= "" then
            local cleanedLeaf = leafName:gsub("%W", "")
            if cleanedLeaf ~= "" then
                local normalizedSuggestion = context.suggestedName and context.suggestedName:lower() or ""
                local containsLeaf = normalizedSuggestion:find(cleanedLeaf:lower(), 1, true) ~= nil
                if normalizedSuggestion == "electrodescript" or normalizedSuggestion == "generatedscript" or normalizedSuggestion == "generatedclient" or not containsLeaf then
                    local suggested = cleanedLeaf
                    if scriptType == "LocalScript" then
                        if not suggested:lower():find("client", 1, true) then
                            suggested = suggested .. "Client"
                        end
                    elseif scriptType == "ModuleScript" then
                        if not suggested:lower():find("module", 1, true) then
                            suggested = suggested .. "Module"
                        end
                    elseif not suggested:lower():find("script", 1, true) then
                        suggested = suggested .. "Script"
                    end
                    if #suggested > 32 then
                        suggested = suggested:sub(1, 32)
                    end
                    context.suggestedName = suggested
                end
            end
        end
    end

    if usesScriptParent and not explicitTargetPath and not allowScriptParentAuto then
        context.reason = "Directory not found for this script. Manual placement required."
        context.requiresManual = true
        context.autoPlaceEnabled = false
        return context
    end

    if not targetPath then
        context.reason = "Directory not found for this script. Manual placement required."
        context.requiresManual = true
        context.autoPlaceEnabled = false
        return context
    end

    local targetParent = resolveTargetPath(targetPath)
    if not targetParent then
        context.reason = "Directory not found for this script. Manual placement required."
        context.requiresManual = true
        context.autoPlaceEnabled = false
        return context
    end

    if targetParent:FindFirstChild(context.suggestedName) then
        context.reason = "A script with this name already exists there. Rename or place manually."
        context.requiresManual = true
        context.autoPlaceEnabled = false
        return context
    end

    context.requiresManual = false
    context.autoPlaceEnabled = true
    if explicitTargetPath then
        context.reason = string.format("Electrode can auto-place this script under %s.", context.targetDisplay)
    else
        context.reason = "Electrode can auto-place this script for you."
    end
    return context
end

-- Removed unused Generator functions to reduce register usage:
-- resetPlacementSection, pasteCodeIntoScript, getCurrentScript, updatePlacementSection

local function closeConfirmationOverlay()
    if confirmationOverlay then
        confirmationOverlay:Destroy()
        confirmationOverlay = nil
    end
end

local function showConfirmationDialog(message, confirmText, cancelText, onConfirm, onCancel)
    closeConfirmationOverlay()

    confirmationOverlay = Instance.new("Frame")
    confirmationOverlay.BackgroundColor3 = Color3.fromRGB(0, 0, 0)
    confirmationOverlay.BackgroundTransparency = 0.4
    confirmationOverlay.Size = UDim2.new(1, 0, 1, 0)
    confirmationOverlay.Position = UDim2.new(0, 0, 0, 0)
    confirmationOverlay.ZIndex = 100
    confirmationOverlay.Parent = MainFrame
    confirmationOverlay.Active = true

    local theme = getActiveTheme()

    local dialogFrame = Instance.new("Frame")
    dialogFrame.Size = UDim2.new(0, 320, 0, 160)
    dialogFrame.Position = UDim2.new(0.5, -160, 0.5, -80)
    dialogFrame.BackgroundTransparency = 1
    dialogFrame.ZIndex = 101
    dialogFrame.Parent = confirmationOverlay
    Style.applyGlassEffect(dialogFrame, {
        transparency = 0.12,
        baseColor = Color3.fromRGB(20, 24, 58),
        cornerRadius = 14,
        gradientRotation = 110
    })

    local dialogPadding = Instance.new("UIPadding")
    dialogPadding.PaddingTop = UDim.new(0, 16)
    dialogPadding.PaddingBottom = UDim.new(0, 16)
    dialogPadding.PaddingLeft = UDim.new(0, 16)
    dialogPadding.PaddingRight = UDim.new(0, 16)
    dialogPadding.Parent = dialogFrame

    local messageLabel = Instance.new("TextLabel")
    messageLabel.Size = UDim2.new(1, 0, 0, 80)
    messageLabel.BackgroundTransparency = 1
    messageLabel.TextWrapped = true
    messageLabel.TextXAlignment = Enum.TextXAlignment.Left
    messageLabel.TextYAlignment = Enum.TextYAlignment.Top
    messageLabel.Font = Enum.Font.Gotham
    messageLabel.TextSize = 14
    messageLabel.TextColor3 = theme.text
    messageLabel.Text = message
    messageLabel.ZIndex = 101
    messageLabel.Parent = dialogFrame

    local dialogButtons = Instance.new("Frame")
    dialogButtons.Size = UDim2.new(1, 0, 0, 40)
    dialogButtons.Position = UDim2.new(0, 0, 1, -50)
    dialogButtons.BackgroundTransparency = 1
    dialogButtons.ZIndex = 101
    dialogButtons.Parent = dialogFrame

    local dialogButtonLayout = Instance.new("UIListLayout")
    dialogButtonLayout.FillDirection = Enum.FillDirection.Horizontal
    dialogButtonLayout.HorizontalAlignment = Enum.HorizontalAlignment.Right
    dialogButtonLayout.Padding = UDim.new(0, 10)
    dialogButtonLayout.Parent = dialogButtons

    local cancelButton = Instance.new("TextButton")
    cancelButton.Size = UDim2.new(0, 110, 0, 32)
    cancelButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    cancelButton.Font = Enum.Font.GothamSemibold
    cancelButton.TextSize = 14
    cancelButton.Text = cancelText or "Cancel"
    cancelButton.ZIndex = 101
    cancelButton.Parent = dialogButtons
    Style.styleSecondaryButton(cancelButton)
    Style.setButtonGradient(
        cancelButton,
        Color3.fromRGB(90, 98, 125),
        Color3.fromRGB(62, 68, 96),
        NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.12),
            NumberSequenceKeypoint.new(1, 0.28)
        })
    )

    local confirmButton = Instance.new("TextButton")
    confirmButton.Size = UDim2.new(0, 140, 0, 32)
    confirmButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    confirmButton.Font = Enum.Font.GothamSemibold
    confirmButton.TextSize = 14
    confirmButton.Text = confirmText or "Confirm"
    confirmButton.ZIndex = 101
    confirmButton.Parent = dialogButtons
    Style.stylePrimaryButton(confirmButton)
    Style.setButtonGradient(
        confirmButton,
        Color3.fromRGB(92, 204, 142),
        Color3.fromRGB(42, 168, 98),
        NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.05),
            NumberSequenceKeypoint.new(1, 0.18)
        })
    )

    cancelButton.MouseButton1Click:Connect(function()
        closeConfirmationOverlay()
        if onCancel then
            onCancel()
        end
    end)

    confirmButton.MouseButton1Click:Connect(function()
        closeConfirmationOverlay()
        if onConfirm then
            onConfirm()
        end
    end)
end

local function performAutoPlacement(context, code)
    if not context or not context.targetPath then
        return false, "Placement path unavailable. Please place manually."
    end

    local targetParent = resolveTargetPath(context.targetPath)
    if not targetParent then
        return false, "Directory not found for this script. Manual placement required."
    end

    if targetParent:FindFirstChild(context.suggestedName) then
        return false, "A script with this name already exists in the target location."
    end

    local scriptClass = context.scriptClass or "Script"
    local newScript
    local success, err = pcall(function()
        newScript = Instance.new(scriptClass)
        newScript.Name = context.suggestedName or scriptClass

        if code and code ~= "" then
            if code:find("-- Generated by Electrode") then
                newScript.Source = code
            else
                newScript.Source = "-- Generated by Electrode\n" .. code
            end
        else
            newScript.Source = "-- Generated by Electrode"
        end

        newScript.Parent = targetParent
    end)

    if not success then
        if newScript then
            newScript:Destroy()
        end
        return false, "Failed to place script: " .. tostring(err)
    end

    ChangeHistoryService:SetWaypoint("Electrode Auto Placement")
    SelectionService:Set({newScript})

    local locationDisplay = context.targetDisplay or table.concat(context.targetPath, " → ")
    return true, string.format("Placed in %s → %s", locationDisplay, newScript.Name)
end

local function describeInstance(instance)
    if not instance then
        return "instance"
    end

    local ok, fullName = pcall(function()
        return instance:GetFullName()
    end)

    if ok and fullName and fullName ~= "" then
        return fullName
    end

    return instance.Name or "instance"
end

local function tryDecodeEnum(enumPath)
    if type(enumPath) ~= "string" then
        return nil
    end

    if not enumPath:find("^Enum%.") then
        return nil
    end

    local parts = string.split(enumPath, ".")
    if #parts < 3 then
        return nil
    end

    local enumType = Enum[parts[2]]
    if not enumType then
        return nil
    end

    local enumItem = enumType[parts[3]]
    return enumItem
end

local function decodeAgentValue(value)
    if type(value) == "string" then
        local enumValue = tryDecodeEnum(value)
        if enumValue then
            return enumValue
        end
        return value
    end

    if type(value) ~= "table" then
        return value
    end

    local valueType = value.type or value.kind or value.__type
    if not valueType then
        return value
    end

    local normalizedType = string.lower(valueType)

    local function toNumber(input, default)
        local numberValue = tonumber(input)
        if numberValue == nil then
            return default
        end
        return numberValue
    end

    if normalizedType == "color3" then
        if value.hex and type(value.hex) == "string" then
            local ok, color = pcall(function()
                return Color3.fromHex(value.hex)
            end)
            if ok then
                return color
            end
        end

        local r = value.r or value.R
        local g = value.g or value.G
        local b = value.b or value.B
        if r and g and b then
            r, g, b = toNumber(r, 0), toNumber(g, 0), toNumber(b, 0)
            if r <= 1 and g <= 1 and b <= 1 then
                return Color3.new(r, g, b)
            else
                return Color3.fromRGB(math.clamp(math.floor(r + 0.5), 0, 255), math.clamp(math.floor(g + 0.5), 0, 255), math.clamp(math.floor(b + 0.5), 0, 255))
            end
        end
    elseif normalizedType == "udim2" then
        local xScale = value.xScale or value.scaleX or (value.x and (value.x.scale or value.xScale)) or 0
        local xOffset = value.xOffset or value.offsetX or (value.x and (value.x.offset or value.xOffset)) or 0
        local yScale = value.yScale or value.scaleY or (value.y and (value.y.scale or value.yScale)) or 0
        local yOffset = value.yOffset or value.offsetY or (value.y and (value.y.offset or value.yOffset)) or 0
        return UDim2.new(
            toNumber(xScale, 0),
            math.floor(toNumber(xOffset, 0) + 0.5),
            toNumber(yScale, 0),
            math.floor(toNumber(yOffset, 0) + 0.5)
        )
    elseif normalizedType == "udim" then
        return UDim.new(
            toNumber(value.scale or value.Scale or 0),
            math.floor(toNumber(value.offset or value.Offset or 0) + 0.5)
        )
    elseif normalizedType == "vector2" then
        return Vector2.new(
            toNumber(value.x or value.X or 0, 0),
            toNumber(value.y or value.Y or 0, 0)
        )
    elseif normalizedType == "vector3" then
        return Vector3.new(
            toNumber(value.x or value.X or 0, 0),
            toNumber(value.y or value.Y or 0, 0),
            toNumber(value.z or value.Z or 0, 0)
        )
    elseif normalizedType == "enum" then
        if value.value then
            return tryDecodeEnum(value.value) or value.value
        end
    elseif normalizedType == "brickcolor" then
        if value.name then
            local ok, brickColor = pcall(function()
                return BrickColor.new(value.name)
            end)
            if ok then
                return brickColor
            end
        elseif value.number then
            local ok, brickColor = pcall(function()
                return BrickColor.new(tonumber(value.number) or 1)
            end)
            if ok then
                return brickColor
            end
        elseif value.r or value.R or value[1] then
            -- Handle r/g/b values - convert to Color3 first, then BrickColor
            local r = value.r or value.R or value[1]
            local g = value.g or value.G or value[2]
            local b = value.b or value.B or value[3]
            if r and g and b then
                local rNum = tonumber(r)
                local gNum = tonumber(g)
                local bNum = tonumber(b)
                if rNum and gNum and bNum then
                    -- Normalize to 0-1 range if values are 0-255
                    if rNum > 1 or gNum > 1 or bNum > 1 then
                        rNum = rNum / 255
                        gNum = gNum / 255
                        bNum = bNum / 255
                    end
                    local color3 = Color3.new(rNum, gNum, bNum)
                    local ok, brickColor = pcall(function()
                        return BrickColor.new(color3)
                    end)
                    if ok then
                        return brickColor
                    end
                end
            end
        end
    elseif normalizedType == "cframe" then
        if value.position then
            local pos = value.position
            local x = toNumber(pos.x or pos.X or pos[1], 0)
            local y = toNumber(pos.y or pos.Y or pos[2], 0)
            local z = toNumber(pos.z or pos.Z or pos[3], 0)
            if value.lookAt then
                local look = value.lookAt
                local lx = toNumber(look.x or look.X or look[1], 0)
                local ly = toNumber(look.y or look.Y or look[2], 0)
                local lz = toNumber(look.z or look.Z or look[3], 0)
                return CFrame.lookAt(Vector3.new(x, y, z), Vector3.new(lx, ly, lz))
            end
            return CFrame.new(x, y, z)
        elseif value.components and type(value.components) == "table" then
            local components = {}
            for index = 1, math.min(#value.components, 12) do
                components[index] = toNumber(value.components[index], 0)
            end
            if #components == 12 then
                local ok, frame = pcall(function()
                    return CFrame.new(table.unpack(components))
                end)
                if ok then
                    return frame
                end
            end
        end
    end

    return value
end

local function trimString(str)
    return (str:gsub("^%s*(.-)%s*$", "%1"))
end

local function tryConvertToColor3(value)
    if typeof(value) == "Color3" then
        return value
    end

    if typeof(value) == "BrickColor" then
        return value.Color
    end

    if type(value) == "table" then
        local r = value.r or value.R or value[1]
        local g = value.g or value.G or value[2]
        local b = value.b or value.B or value[3]
        if r and g and b then
            r, g, b = tonumber(r) or 0, tonumber(g) or 0, tonumber(b) or 0
            if r <= 1 and g <= 1 and b <= 1 then
                return Color3.new(r, g, b)
            end
            return Color3.fromRGB(math.clamp(math.floor(r + 0.5), 0, 255), math.clamp(math.floor(g + 0.5), 0, 255), math.clamp(math.floor(b + 0.5), 0, 255))
        end

        if value.hex and type(value.hex) == "string" then
            local ok, clr = pcall(function()
                return Color3.fromHex(value.hex)
            end)
            if ok then
                return clr
            end
        end
    end

    if type(value) ~= "string" then
        return nil
    end

    local str = trimString(value)
    if str == "" then
        return nil
    end

    if str:match("^#%x%x%x%x%x%x$") or str:match("^#%x%x%x$") then
        local ok, clr = pcall(function()
            return Color3.fromHex(str)
        end)
        if ok then
            return clr
        end
    end

    local r, g, b = str:match("^rgb?%s*%(%s*([%d%.]+)%s*,%s*([%d%.]+)%s*,%s*([%d%.]+)%s*%)$")
    if r and g and b then
        r, g, b = tonumber(r) or 0, tonumber(g) or 0, tonumber(b) or 0
        if r <= 1 and g <= 1 and b <= 1 then
            return Color3.new(r, g, b)
        end
        return Color3.fromRGB(math.clamp(math.floor(r + 0.5), 0, 255), math.clamp(math.floor(g + 0.5), 0, 255), math.clamp(math.floor(b + 0.5), 0, 255))
    end

    r, g, b = str:match("^([%d%.]+)%s*,%s*([%d%.]+)%s*,%s*([%d%.]+)$")
    if r and g and b then
        r, g, b = tonumber(r) or 0, tonumber(g) or 0, tonumber(b) or 0
        if r <= 1 and g <= 1 and b <= 1 then
            return Color3.new(r, g, b)
        end
        return Color3.fromRGB(math.clamp(math.floor(r + 0.5), 0, 255), math.clamp(math.floor(g + 0.5), 0, 255), math.clamp(math.floor(b + 0.5), 0, 255))
    end

    local okBrick, brickColor = pcall(function()
        return BrickColor.new(str)
    end)
    if okBrick then
        return brickColor.Color
    end

    return nil
end

local function tryConvertToBrickColor(value)
    if typeof(value) == "BrickColor" then
        return value
    end

    if typeof(value) == "Color3" then
        local ok, brick = pcall(function()
            return BrickColor.new(value)
        end)
        if ok then
            return brick
        end
    end

    if type(value) == "number" then
        local ok, brick = pcall(function()
            return BrickColor.new(value)
        end)
        if ok then
            return brick
        end
    end

    if type(value) == "string" then
        local ok, brick = pcall(function()
            return BrickColor.new(trimString(value))
        end)
        if ok then
            return brick
        end
    end

    -- Handle table with r/g/b or R/G/B values (convert to Color3 first, then BrickColor)
    if type(value) == "table" then
        local r = value.r or value.R or value[1]
        local g = value.g or value.G or value[2]
        local b = value.b or value.B or value[3]
        if r and g and b then
            local rNum = tonumber(r)
            local gNum = tonumber(g)
            local bNum = tonumber(b)
            if rNum and gNum and bNum then
                -- Normalize to 0-1 range if values are 0-255
                if rNum > 1 or gNum > 1 or bNum > 1 then
                    rNum = rNum / 255
                    gNum = gNum / 255
                    bNum = bNum / 255
                end
                local color3 = Color3.new(rNum, gNum, bNum)
                local ok, brick = pcall(function()
                    return BrickColor.new(color3)
                end)
                if ok then
                    return brick
                end
            end
        end
        -- Try as BrickColor name if table has a name field
        if value.name and type(value.name) == "string" then
            local ok, brick = pcall(function()
                return BrickColor.new(value.name)
            end)
            if ok then
                return brick
            end
        end
        -- Try as BrickColor number if table has a number field
        if value.number then
            local num = tonumber(value.number)
            if num then
                local ok, brick = pcall(function()
                    return BrickColor.new(num)
                end)
                if ok then
                    return brick
                end
            end
        end
    end

    return nil
end

local function tryConvertToEnumItem(enumType, value)
    if typeof(value) == "EnumItem" then
        if value.EnumType == enumType then
            return value
        end
        return nil
    end

    if type(value) ~= "string" then
        return nil
    end

    local trimmed = trimString(value)
    if trimmed == "" then
        return nil
    end

    local direct = enumType[trimmed]
    if direct then
        return direct
    end

    local normalized = trimmed:gsub("[%s%-_]", "")
    direct = enumType[normalized]
    if direct then
        return direct
    end

    local upper = trimmed:upper()
    direct = enumType[upper]
    if direct then
        return direct
    end

    for _, enumItem in ipairs(enumType:GetEnumItems()) do
        if string.lower(enumItem.Name) == string.lower(trimmed) then
            return enumItem
        end
    end

    return nil
end

local function coerceValueForProperty(instance, property, rawValue, resolvedValue)
    local okCurrent, currentValue = pcall(function()
        return instance[property]
    end)

    local propertyType = nil
    if okCurrent then
        propertyType = typeof(currentValue)
    end

    local candidates = { resolvedValue, rawValue }

    if propertyType == "Color3" then
        for _, candidate in ipairs(candidates) do
            local converted = tryConvertToColor3(candidate)
            if converted then
                return converted
            end
        end
    elseif propertyType == "BrickColor" then
        for _, candidate in ipairs(candidates) do
            local converted = tryConvertToBrickColor(candidate)
            if converted then
                return converted
            end
        end
    elseif propertyType == "EnumItem" then
        local enumType = currentValue.EnumType
        for _, candidate in ipairs(candidates) do
            local converted = tryConvertToEnumItem(enumType, candidate)
            if converted then
                return converted
            end
        end
    elseif propertyType == "boolean" then
        for _, candidate in ipairs(candidates) do
            if type(candidate) == "string" then
                local lowered = string.lower(trimString(candidate))
                if lowered == "true" or lowered == "yes" or lowered == "1" then
                    return true
                elseif lowered == "false" or lowered == "no" or lowered == "0" then
                    return false
                end
            end
        end
    elseif propertyType == "number" then
        for _, candidate in ipairs(candidates) do
            local numValue = tonumber(candidate)
            if numValue then
                return numValue
            end
        end
    elseif propertyType == "string" then
        for _, candidate in ipairs(candidates) do
            if type(candidate) == "string" then
                return candidate
            elseif type(candidate) == "number" or type(candidate) == "boolean" then
                return tostring(candidate)
            end
        end
    else
        -- Property type unknown or uninitialized - check common string properties
        local commonStringProperties = {
            "Text", "Name", "TextLabel", "PlaceholderText", "Value", 
            "Hint", "ToolTip", "Description", "Title", "Caption"
        }
        for _, commonProp in ipairs(commonStringProperties) do
            if property == commonProp then
                for _, candidate in ipairs(candidates) do
                    if type(candidate) == "string" then
                        return candidate
                    elseif type(candidate) == "number" or type(candidate) == "boolean" then
                        return tostring(candidate)
                    end
                end
                break
            end
        end
    end

    return nil
end

local function setInstanceProperty(instance, property, rawValue)
    -- Special handling for padding property on UI elements
    -- Padding must be set via a UIPadding child instance, not as a direct property
    if (property == "padding" or property == "Padding") and instance:IsA("GuiObject") then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- Try to find existing UIPadding child
        local uiPadding = instance:FindFirstChildOfClass("UIPadding")
        if not uiPadding then
            -- Create new UIPadding instance
            uiPadding = Instance.new("UIPadding")
            uiPadding.Parent = instance
        end
        
        -- Handle different padding value formats
        if type(resolvedValue) == "table" then
            -- Table format: {top=10, bottom=10, left=10, right=10} or {all=10}
            if resolvedValue.all then
                local allValue = tonumber(resolvedValue.all) or 0
                uiPadding.PaddingTop = UDim.new(0, allValue)
                uiPadding.PaddingBottom = UDim.new(0, allValue)
                uiPadding.PaddingLeft = UDim.new(0, allValue)
                uiPadding.PaddingRight = UDim.new(0, allValue)
            else
                if resolvedValue.top or resolvedValue.Top then
                    uiPadding.PaddingTop = UDim.new(0, tonumber(resolvedValue.top or resolvedValue.Top) or 0)
                end
                if resolvedValue.bottom or resolvedValue.Bottom then
                    uiPadding.PaddingBottom = UDim.new(0, tonumber(resolvedValue.bottom or resolvedValue.Bottom) or 0)
                end
                if resolvedValue.left or resolvedValue.Left then
                    uiPadding.PaddingLeft = UDim.new(0, tonumber(resolvedValue.left or resolvedValue.Left) or 0)
                end
                if resolvedValue.right or resolvedValue.Right then
                    uiPadding.PaddingRight = UDim.new(0, tonumber(resolvedValue.right or resolvedValue.Right) or 0)
                end
            end
        elseif type(resolvedValue) == "number" then
            -- Single number - apply to all sides
            uiPadding.PaddingTop = UDim.new(0, resolvedValue)
            uiPadding.PaddingBottom = UDim.new(0, resolvedValue)
            uiPadding.PaddingLeft = UDim.new(0, resolvedValue)
            uiPadding.PaddingRight = UDim.new(0, resolvedValue)
        else
            -- Try to convert to number
            local numValue = tonumber(resolvedValue) or tonumber(rawValue) or 0
            uiPadding.PaddingTop = UDim.new(0, numValue)
            uiPadding.PaddingBottom = UDim.new(0, numValue)
            uiPadding.PaddingLeft = UDim.new(0, numValue)
            uiPadding.PaddingRight = UDim.new(0, numValue)
        end
        
        return true
    end
    
    -- Special handling for PrimaryPart property on Model instances
    if property == "PrimaryPart" or property == "primaryPart" then
        -- PrimaryPart only exists on Model instances, not Folders or other types
        if not instance:IsA("Model") then
            return false, string.format("Failed to set PrimaryPart on %s: PrimaryPart is only available on Model instances, not %s", describeInstance(instance), instance.ClassName)
        end
        
        local partName = nil
        if type(rawValue) == "string" then
            partName = rawValue
        elseif type(rawValue) == "table" and (rawValue.name or rawValue.Name or rawValue.path or rawValue.Path) then
            partName = rawValue.name or rawValue.Name or rawValue.path or rawValue.Path
        end
        
        if partName then
            -- If partName is a full path, extract just the name
            if partName:find("%.") then
                local segments = {}
                for segment in partName:gmatch("[^%.]+") do
                    table.insert(segments, segment)
                end
                if #segments > 0 then
                    partName = segments[#segments]
                end
            end
            
            -- Find the child part by name
            local part = instance:FindFirstChild(partName, true)
            if part and part:IsA("BasePart") then
                local success, err = pcall(function()
                    instance.PrimaryPart = part
                end)
                if success then
                    return true
                end
                return false, string.format("Failed to set PrimaryPart on %s: %s", describeInstance(instance), tostring(err))
            else
                -- Try finding any BasePart child if exact name not found
                for _, child in ipairs(instance:GetDescendants()) do
                    if child:IsA("BasePart") then
                        local success, err = pcall(function()
                            instance.PrimaryPart = child
                        end)
                        if success then
                            return true
                        end
                    end
                end
                return false, string.format("Failed to set PrimaryPart on %s: No BasePart found with name '%s'", describeInstance(instance), partName)
            end
        end
    end
    
    -- Special handling for CFrame properties (CFrame, C1, C0) - convert tables to CFrame objects
    local cframeProperties = {"CFrame", "C1", "C0", "cframe", "c1", "c0"}
    local isCFrameProperty = false
    for _, cframeProp in ipairs(cframeProperties) do
        if property == cframeProp then
            isCFrameProperty = true
            break
        end
    end
    
    if isCFrameProperty then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already a CFrame, use it directly
        if typeof(resolvedValue) == "CFrame" then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to convert table to CFrame
        if type(resolvedValue) == "table" then
            local cframe = nil
            
            -- Try CFrame.new() with 12-component format: {X, Y, Z, R00, R01, R02, R10, R11, R12, R20, R21, R22}
            local nums = {}
            local count = 0
            for i = 1, 12 do
                local val = resolvedValue[i] or resolvedValue[tostring(i)]
                if val then
                    local num = tonumber(val)
                    if num then
                        nums[i] = num
                        count = count + 1
                    end
                end
            end
            
            if count == 12 then
                -- 12-component CFrame (position + rotation matrix)
                local success, result = pcall(function()
                    return CFrame.new(nums[1], nums[2], nums[3], nums[4], nums[5], nums[6], nums[7], nums[8], nums[9], nums[10], nums[11], nums[12])
                end)
                if success then
                    cframe = result
                end
            elseif count == 6 then
                -- 6-component CFrame (position + look direction)
                local success, result = pcall(function()
                    return CFrame.new(nums[1], nums[2], nums[3], nums[4], nums[5], nums[6])
                end)
                if success then
                    cframe = result
                end
            elseif count == 3 then
                -- 3-component CFrame (position only)
                local success, result = pcall(function()
                    return CFrame.new(nums[1], nums[2], nums[3])
                end)
                if success then
                    cframe = result
                end
            end
            
            -- Try position + quaternion format: {x, y, z, qx, qy, qz, qw}
            if not cframe and (resolvedValue.x or resolvedValue.X) then
                local x = tonumber(resolvedValue.x or resolvedValue.X) or 0
                local y = tonumber(resolvedValue.y or resolvedValue.Y) or 0
                local z = tonumber(resolvedValue.z or resolvedValue.Z) or 0
                
                if resolvedValue.qx or resolvedValue.QX then
                    -- Quaternion format
                    local qx = tonumber(resolvedValue.qx or resolvedValue.QX) or 0
                    local qy = tonumber(resolvedValue.qy or resolvedValue.QY) or 0
                    local qz = tonumber(resolvedValue.qz or resolvedValue.QZ) or 0
                    local qw = tonumber(resolvedValue.qw or resolvedValue.QW) or 1
                    local success, result = pcall(function()
                        return CFrame.new(x, y, z, qx, qy, qz, qw)
                    end)
                    if success then
                        cframe = result
                    end
                else
                    -- Position only
                    local success, result = pcall(function()
                        return CFrame.new(x, y, z)
                    end)
                    if success then
                        cframe = result
                    end
                end
            end
            
            -- Try Position + LookAt format
            if not cframe and resolvedValue.Position and resolvedValue.LookAt then
                local pos = resolvedValue.Position
                local look = resolvedValue.LookAt
                local px, py, pz = tonumber(pos.x or pos.X) or 0, tonumber(pos.y or pos.Y) or 0, tonumber(pos.z or pos.Z) or 0
                local lx, ly, lz = tonumber(look.x or look.X) or 0, tonumber(look.y or look.Y) or 0, tonumber(look.z or look.Z) or 0
                local success, result = pcall(function()
                    return CFrame.lookAt(Vector3.new(px, py, pz), Vector3.new(lx, ly, lz))
                end)
                if success then
                    cframe = result
                end
            end
            
            if cframe then
                local success, err = pcall(function()
                    instance[property] = cframe
                end)
                if success then
                    return true
                end
                return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
            else
                -- Couldn't convert - return helpful error
                return false, string.format("Failed to set %s on %s: Unable to convert table to CFrame. Expected format: array of 3/6/12 numbers, or {x, y, z} with optional rotation", property, describeInstance(instance))
            end
        else
            -- Not a table and not a CFrame - this is invalid for CFrame properties
            return false, string.format("Failed to set %s on %s: Expected CFrame or table convertible to CFrame, got %s", property, describeInstance(instance), typeof(resolvedValue))
        end
        
        -- This should never be reached, but just in case
        return false, string.format("Failed to set %s on %s: Unexpected error processing CFrame value", property, describeInstance(instance))
    end
    
    -- Special handling for Part0/Part1 properties (WeldConstraint, Motor6D, etc.) - resolve string paths to instances
    local partProperties = {"Part0", "Part1", "part0", "part1"}
    local isPartProperty = false
    for _, partProp in ipairs(partProperties) do
        if property == partProp then
            isPartProperty = true
            break
        end
    end
    
    if isPartProperty then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already an instance (BasePart), use it directly
        if typeof(resolvedValue) == "Instance" and resolvedValue:IsA("BasePart") then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to resolve string path to instance
        local partPath = nil
        if type(resolvedValue) == "string" then
            partPath = resolvedValue
        elseif type(resolvedValue) == "table" and (resolvedValue.path or resolvedValue.Path or resolvedValue.name or resolvedValue.Name) then
            partPath = resolvedValue.path or resolvedValue.Path or resolvedValue.name or resolvedValue.Name
        elseif type(rawValue) == "string" then
            partPath = rawValue
        end
        
        if partPath then
            -- Remove "game." or "Game." prefix if present for cleaner path handling
            partPath = partPath:gsub("^[Gg]ame%.", "")
            
            -- Try resolving as absolute path first (e.g., "Workspace.Guns.Pistol.Handle")
            local partInstance, pathErr = resolveInstanceFromPath(partPath, {})
            
            -- If absolute path fails, try relative paths from the current instance's location
            if not partInstance or not partInstance:IsA("BasePart") then
                -- Strategy 1: Try finding relative to the weld's parent (common case: parts are siblings)
                local currentParent = instance.Parent
                if currentParent then
                    -- Extract just the part name from the path
                    local partName = partPath
                    if partPath:find("%.") then
                        local segments = {}
                        for segment in partPath:gmatch("[^%.]+") do
                            table.insert(segments, segment)
                        end
                        if #segments > 0 then
                            partName = segments[#segments]
                        end
                    end
                    
                    -- Try finding in the same parent (sibling parts)
                    partInstance = currentParent:FindFirstChild(partName, true)
                    if partInstance and partInstance:IsA("BasePart") then
                        -- Found sibling!
                    end
                    
                    -- Try finding in parent's parent (common for welds on parts)
                    if not partInstance and currentParent.Parent then
                        partInstance = currentParent.Parent:FindFirstChild(partName, true)
                        if partInstance and partInstance:IsA("BasePart") then
                            -- Found in parent's parent!
                        end
                    end
                    
                    -- Try constructing full path from parent
                    if not partInstance then
                        local parentPath = currentParent:GetFullName()
                        local fullPath = parentPath .. "." .. partPath
                        partInstance, _ = resolveInstanceFromPath(fullPath, {})
                    end
                end
                
                -- Strategy 2: Try searching from the current instance's model
                if not partInstance or not partInstance:IsA("BasePart") then
                    local model = instance:FindFirstAncestorOfClass("Model")
                    if model then
                        -- Extract part name from path
                        local partName = partPath
                        if partPath:find("%.") then
                            local segments = {}
                            for segment in partPath:gmatch("[^%.]+") do
                                table.insert(segments, segment)
                            end
                            if #segments > 0 then
                                partName = segments[#segments]
                            end
                        end
                        
                        partInstance = model:FindFirstChild(partName, true)
                        if partInstance and partInstance:IsA("BasePart") then
                            -- Found it in model!
                        end
                    end
                end
                
                -- Strategy 3: Try resolving the full path from workspace
                if not partInstance or not partInstance:IsA("BasePart") then
                    -- Ensure path starts with Workspace
                    local fullPath = partPath
                    if not partPath:match("^[Ww]orkspace%.") then
                        fullPath = "Workspace." .. partPath
                    end
                    partInstance, _ = resolveInstanceFromPath(fullPath, {})
                end
                
                -- Strategy 4: Last resort - search workspace for the part name
                if not partInstance or not partInstance:IsA("BasePart") then
                    -- Extract just the part name (last segment of path)
                    local partName = partPath
                    if partPath:find("%.") then
                        local segments = {}
                        for segment in partPath:gmatch("[^%.]+") do
                            table.insert(segments, segment)
                        end
                        if #segments > 0 then
                            partName = segments[#segments]
                        end
                    end
                    
                    -- Search workspace for a part with this name
                    local foundParts = {}
                    for _, descendant in ipairs(workspace:GetDescendants()) do
                        if descendant:IsA("BasePart") and descendant.Name == partName then
                            table.insert(foundParts, descendant)
                        end
                    end
                    
                    -- Use the part if exactly one was found
                    if #foundParts == 1 then
                        partInstance = foundParts[1]
                    end
                end
            end
            
            if partInstance and partInstance:IsA("BasePart") then
                local success, err = pcall(function()
                    instance[property] = partInstance
                end)
                if success then
                    return true
                end
                return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
            else
                return false, string.format("Failed to set %s on %s: Could not resolve path '%s' to a BasePart instance", property, describeInstance(instance), partPath)
            end
        else
            return false, string.format("Failed to set %s on %s: Expected string path or BasePart instance, got %s", property, describeInstance(instance), typeof(resolvedValue))
        end
    end
    
    -- Special handling for Vector3 properties (Size, Position on BaseParts, Velocity, etc.) - convert tables to Vector3 objects
    -- Note: Size/Position on GUI elements are UDim2, handled separately below
    local vector3Properties = {"Velocity", "velocity", "RotVelocity", "rotVelocity", "AssemblyLinearVelocity", "assemblyLinearVelocity", "AssemblyAngularVelocity", "assemblyAngularVelocity"}
    local isVector3Property = false
    -- Only treat Size/Position as Vector3 if instance is NOT a GUI element
    if not (instance:IsA("GuiObject") or instance:IsA("UIBase")) then
        if property == "Size" or property == "size" or property == "Position" or property == "position" then
            isVector3Property = true
        end
    end
    -- Check other Vector3 properties
    if not isVector3Property then
        for _, vec3Prop in ipairs(vector3Properties) do
            if property == vec3Prop then
                isVector3Property = true
                break
            end
        end
    end
    
    if isVector3Property then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already a Vector3, use it directly
        if typeof(resolvedValue) == "Vector3" then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to convert table to Vector3
        if type(resolvedValue) == "table" then
            local x, y, z = nil, nil, nil
            
            -- Try array format: {1, 2, 3} or {"1", "2", "3"}
            if resolvedValue[1] or resolvedValue["1"] then
                x = tonumber(resolvedValue[1] or resolvedValue["1"])
                y = tonumber(resolvedValue[2] or resolvedValue["2"])
                z = tonumber(resolvedValue[3] or resolvedValue["3"])
            -- Try object format: {x: 1, y: 2, z: 3} or {X: 1, Y: 2, Z: 3}
            elseif resolvedValue.x or resolvedValue.X then
                x = tonumber(resolvedValue.x or resolvedValue.X)
                y = tonumber(resolvedValue.y or resolvedValue.Y)
                z = tonumber(resolvedValue.z or resolvedValue.Z)
            end
            
            if x and y and z then
                local success, result = pcall(function()
                    return Vector3.new(x, y, z)
                end)
                if success and result then
                    local assignSuccess, assignErr = pcall(function()
                        instance[property] = result
                    end)
                    if assignSuccess then
                        return true
                    end
                    return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(assignErr))
                end
            end
            
            return false, string.format("Failed to set %s on %s: Unable to convert table to Vector3. Expected format: {x, y, z} or {X, Y, Z} or array [x, y, z]", property, describeInstance(instance))
        else
            return false, string.format("Failed to set %s on %s: Expected Vector3 or table convertible to Vector3, got %s", property, describeInstance(instance), typeof(resolvedValue))
        end
    end
    
    -- Special handling for Vector2 properties (AnchorPoint, etc.) - convert tables to Vector2 objects
    local vector2Properties = {"AnchorPoint", "anchorPoint", "anchorpoint"}
    local isVector2Property = false
    for _, vec2Prop in ipairs(vector2Properties) do
        if property == vec2Prop then
            isVector2Property = true
            break
        end
    end
    
    if isVector2Property then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already a Vector2, use it directly
        if typeof(resolvedValue) == "Vector2" then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to convert table to Vector2
        if type(resolvedValue) == "table" then
            local x, y = nil, nil
            
            -- Try array format: {1, 2} or {"1", "2"}
            if resolvedValue[1] or resolvedValue["1"] then
                x = tonumber(resolvedValue[1] or resolvedValue["1"])
                y = tonumber(resolvedValue[2] or resolvedValue["2"])
            -- Try object format: {x: 1, y: 2} or {X: 1, Y: 2}
            elseif resolvedValue.x or resolvedValue.X then
                x = tonumber(resolvedValue.x or resolvedValue.X)
                y = tonumber(resolvedValue.y or resolvedValue.Y)
            end
            
            if x and y then
                local success, result = pcall(function()
                    return Vector2.new(x, y)
                end)
                if success and result then
                    local assignSuccess, assignErr = pcall(function()
                        instance[property] = result
                    end)
                    if assignSuccess then
                        return true
                    end
                    return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(assignErr))
                end
            end
            
            return false, string.format("Failed to set %s on %s: Unable to convert table to Vector2. Expected format: {x, y} or {X, Y} or array [x, y]", property, describeInstance(instance))
        else
            return false, string.format("Failed to set %s on %s: Expected Vector2 or table convertible to Vector2, got %s", property, describeInstance(instance), typeof(resolvedValue))
        end
    end
    
    -- Special handling for UDim2 properties (Size, Position on GUI elements) - convert tables to UDim2 objects
    local udim2Properties = {"Size", "Position", "size", "position"}
    local isUDim2Property = false
    -- Only check for UDim2 if instance is a GUI element
    if instance:IsA("GuiObject") or instance:IsA("UIBase") then
        for _, udim2Prop in ipairs(udim2Properties) do
            if property == udim2Prop then
                isUDim2Property = true
                break
            end
        end
    end
    
    if isUDim2Property then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already a UDim2, use it directly
        if typeof(resolvedValue) == "UDim2" then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to convert table to UDim2
        if type(resolvedValue) == "table" then
            local udim2 = nil
            
            -- Try UDim2 format: {X = {Scale = 0.5, Offset = 0}, Y = {Scale = 0.5, Offset = 0}}
            if resolvedValue.X and resolvedValue.Y then
                local xScale = tonumber(resolvedValue.X.Scale or resolvedValue.X.scale) or 0
                local xOffset = tonumber(resolvedValue.X.Offset or resolvedValue.X.offset) or 0
                local yScale = tonumber(resolvedValue.Y.Scale or resolvedValue.Y.scale) or 0
                local yOffset = tonumber(resolvedValue.Y.Offset or resolvedValue.Y.offset) or 0
                
                local success, result = pcall(function()
                    return UDim2.new(xScale, xOffset, yScale, yOffset)
                end)
                if success then
                    udim2 = result
                end
            -- Try array format: {scaleX, offsetX, scaleY, offsetY}
            elseif resolvedValue[1] or resolvedValue["1"] then
                local scaleX = tonumber(resolvedValue[1] or resolvedValue["1"]) or 0
                local offsetX = tonumber(resolvedValue[2] or resolvedValue["2"]) or 0
                local scaleY = tonumber(resolvedValue[3] or resolvedValue["3"]) or 0
                local offsetY = tonumber(resolvedValue[4] or resolvedValue["4"]) or 0
                
                local success, result = pcall(function()
                    return UDim2.new(scaleX, offsetX, scaleY, offsetY)
                end)
                if success then
                    udim2 = result
                end
            -- Try simple format: {x = {scale, offset}, y = {scale, offset}}
            elseif resolvedValue.x and resolvedValue.y then
                local xVals = resolvedValue.x
                local yVals = resolvedValue.y
                local xScale = tonumber(xVals[1] or xVals.scale or xVals.Scale) or 0
                local xOffset = tonumber(xVals[2] or xVals.offset or xVals.Offset) or 0
                local yScale = tonumber(yVals[1] or yVals.scale or yVals.Scale) or 0
                local yOffset = tonumber(yVals[2] or yVals.offset or yVals.Offset) or 0
                
                local success, result = pcall(function()
                    return UDim2.new(xScale, xOffset, yScale, yOffset)
                end)
                if success then
                    udim2 = result
                end
            end
            
            if udim2 then
                local success, err = pcall(function()
                    instance[property] = udim2
                end)
                if success then
                    return true
                end
                return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
            else
                return false, string.format("Failed to set %s on %s: Unable to convert table to UDim2. Expected format: {X = {Scale, Offset}, Y = {Scale, Offset}} or [scaleX, offsetX, scaleY, offsetY]", property, describeInstance(instance))
            end
        else
            return false, string.format("Failed to set %s on %s: Expected UDim2 or table convertible to UDim2, got %s", property, describeInstance(instance), typeof(resolvedValue))
        end
    end
    
    -- Special handling for UDim properties (used in padding, etc.) - convert tables to UDim objects
    local udimProperties = {"PaddingTop", "PaddingBottom", "PaddingLeft", "PaddingRight", "paddingTop", "paddingBottom", "paddingLeft", "paddingRight"}
    local isUDimProperty = false
    for _, udimProp in ipairs(udimProperties) do
        if property == udimProp then
            isUDimProperty = true
            break
        end
    end
    
    if isUDimProperty then
        local resolvedValue = decodeAgentValue(rawValue)
        
        -- If already a UDim, use it directly
        if typeof(resolvedValue) == "UDim" then
            local success, err = pcall(function()
                instance[property] = resolvedValue
            end)
            if success then
                return true
            end
            return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
        end
        
        -- Try to convert table or number to UDim
        if type(resolvedValue) == "table" then
            local scale = tonumber(resolvedValue.Scale or resolvedValue.scale or resolvedValue[1] or resolvedValue["1"]) or 0
            local offset = tonumber(resolvedValue.Offset or resolvedValue.offset or resolvedValue[2] or resolvedValue["2"]) or 0
            
            local success, result = pcall(function()
                return UDim.new(scale, offset)
            end)
            if success and result then
                local assignSuccess, assignErr = pcall(function()
                    instance[property] = result
                end)
                if assignSuccess then
                    return true
                end
                return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(assignErr))
            end
        elseif type(resolvedValue) == "number" or type(rawValue) == "number" then
            -- Single number means offset with scale 0
            local offset = tonumber(resolvedValue) or tonumber(rawValue) or 0
            local success, result = pcall(function()
                return UDim.new(0, offset)
            end)
            if success and result then
                local assignSuccess, assignErr = pcall(function()
                    instance[property] = result
                end)
                if assignSuccess then
                    return true
                end
            end
        end
        
        return false, string.format("Failed to set %s on %s: Unable to convert to UDim. Expected format: {Scale, Offset} or number (offset only)", property, describeInstance(instance))
    end
    
    local resolvedValue = decodeAgentValue(rawValue)
    
    -- Proactively check for Text properties and coerce boolean/number to string
    local commonStringProperties = {
        "Text", "Name", "TextLabel", "PlaceholderText", "Value", 
        "Hint", "ToolTip", "Description", "Title", "Caption"
    }
    for _, commonProp in ipairs(commonStringProperties) do
        if property == commonProp then
            if type(resolvedValue) == "boolean" or type(resolvedValue) == "number" then
                resolvedValue = tostring(resolvedValue)
            elseif type(rawValue) == "boolean" or type(rawValue) == "number" then
                resolvedValue = tostring(rawValue)
            end
            break
        end
    end
    
    -- Proactively check for BrickColor properties and convert tables/Color3 to BrickColor
    if property == "BrickColor" or property == "brickColor" then
        if type(resolvedValue) == "table" or typeof(resolvedValue) == "Color3" then
            local converted = tryConvertToBrickColor(resolvedValue)
            if converted then
                resolvedValue = converted
            elseif type(rawValue) == "table" then
                converted = tryConvertToBrickColor(rawValue)
                if converted then
                    resolvedValue = converted
                end
            end
        end
    end
    
    local function tryAssign(value)
        return pcall(function()
            instance[property] = value
        end)
    end

    local success, err = tryAssign(resolvedValue)

    if success then
        return true
    end

    local coerced = coerceValueForProperty(instance, property, rawValue, resolvedValue)
    if coerced ~= nil then
        local retrySuccess, retryErr = tryAssign(coerced)
        if retrySuccess then
            return true
        end
        err = retryErr
    end

    if err and type(err) == "string" then
        local lowered = string.lower(err)
        if string.find(lowered, "string expected", 1, true) then
            local fallbackSource = resolvedValue
            if type(fallbackSource) ~= "string" then
                if fallbackSource == nil then
                    fallbackSource = rawValue
                end
                local fallbackType = type(fallbackSource)
                if fallbackType == "boolean" or fallbackType == "number" then
                    local fallbackSuccess, fallbackErr = tryAssign(tostring(fallbackSource))
                    if fallbackSuccess then
                        return true
                    end
                    err = fallbackErr
                end
            end
        end
    end

    return false, string.format("Failed to set %s on %s: %s", property, describeInstance(instance), tostring(err))
end

local function applyProperties(instance, properties)
    if type(properties) ~= "table" then
        return false, "Properties payload must be a JSON object."
    end

    for property, rawValue in pairs(properties) do
        local success, err = setInstanceProperty(instance, property, rawValue)
        if not success then
            return false, err
        end
    end

    return true
end

local function ensureInstance(action)
    if not action or not action.path then
        return false, "Missing path for ensure_instance action"
    end

    local leafClass = action.class or "Folder"
    local segments = splitPathString(action.path)
    if #segments == 0 then
        return false, "Invalid path supplied"
    end

    local instance, err = resolveInstanceFromSegments(segments, {
        createMissing = true,
        createLeaf = true,
        leafClass = leafClass,
        intermediateClass = "Folder"
    })

    if not instance then
        return false, err or ("Failed to ensure instance at " .. action.path)
    end

    -- Verify the instance was actually created and is accessible
    -- This is especially important for folders that will be used by subsequent actions
    if not instance.Parent then
        return false, "Created instance has no parent: " .. action.path
    end
    
    -- Verify the instance is immediately accessible via FindFirstChild
    local verifyCheck = instance.Parent:FindFirstChild(instance.Name)
    if not verifyCheck or verifyCheck ~= instance then
        -- Instance might not be immediately accessible - this can happen in Studio
        -- Force Studio Explorer to refresh by briefly selecting the instance
        -- This helps Studio recognize newly created instances
        pcall(function()
            SelectionService:Set({instance})
            task.defer(function()
                SelectionService:Set({})
            end)
        end)
    end
    
    -- Final check: ensure instance has valid Parent
    if not instance.Parent then
        return false, "Instance lost its parent immediately after creation at " .. action.path
    end

    if action.properties then
        local success, propErr = applyProperties(instance, action.properties)
        if not success then
            return false, propErr
        end
    end

    return true, instance
end

local function writeScriptSource(action)
    if not action or not action.path or not action.source then
        return false, "Missing path or source for write_script action"
    end

    local segments = splitPathString(action.path)
    if #segments < 2 then
        return false, "Script path must include parent service and script name"
    end

    local scriptName = table.remove(segments)
    local parent, err = resolveInstanceFromSegments(segments, {
        createMissing = true,
        intermediateClass = "Folder"
    })

    if not parent then
        -- Provide more detailed error information
        local parentPathStr = table.concat(segments, "/")
        local errorMsg = err or ("Cannot find parent for script path: " .. action.path)
        errorMsg = errorMsg .. " (Parent path: " .. parentPathStr .. ")"
        
        -- Try to see if we can identify what's missing
        if #segments > 0 then
            local firstSegment = segments[1]
            local firstExists = false
            pcall(function()
                local test = game:GetService(firstSegment)
                if test then firstExists = true end
            end)
            if not firstExists then
                pcall(function()
                    local test = game:FindFirstChild(firstSegment)
                    if test then firstExists = true end
                end)
            end
            if not firstExists then
                errorMsg = errorMsg .. " (First segment '" .. firstSegment .. "' not found)"
            end
        end
        
        return false, errorMsg
    end

    local existing = parent:FindFirstChild(scriptName)
    local scriptClass = action.class or "Script"
    local targetScript = existing

    if existing then
        if not existing:IsA("LuaSourceContainer") then
            if action.overwrite ~= false then
                existing:Destroy()
                existing = nil
            else
                return false, string.format("Instance '%s' exists at %s but is not a script", scriptName, action.path)
            end
        elseif existing.ClassName ~= scriptClass and action.overwrite ~= false then
            existing:Destroy()
            existing = nil
        end
    end

    if not existing then
        local successCreate, newScript = pcall(function()
            local inst = Instance.new(scriptClass)
            inst.Name = scriptName
            inst.Parent = parent
            return inst
        end)

        if not successCreate then
            return false, "Failed to create script " .. scriptName
        end

        targetScript = newScript
    else
        targetScript = existing
    end

    if action.tags and CollectionService then
        for _, tag in ipairs(action.tags) do
            CollectionService:AddTag(targetScript, tag)
        end
    end

    local header = action.header or "-- Generated by Electrode Agent"
    local source = action.source
    if action.includeHeader ~= false and not source:find(header, 1, true) then
        source = header .. "\n" .. source
    end

    local successWrite, writeErr = pcall(function()
        targetScript.Source = source
    end)

    if not successWrite then
        return false, "Failed to set script source: " .. tostring(writeErr)
    end

    return true, targetScript
end

local function setProperty(action)
    if not action or not action.path or not action.property then
        return false, "Invalid set_property action"
    end

    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = action.createMissing,
        createLeaf = action.createMissing and action.leafClass ~= nil,
        leafClass = action.leafClass,
        intermediateClass = action.intermediateClass or "Folder"
    })

    if not instance then
        return false, err or "Unable to resolve path for property set"
    end

    if action.properties then
        local success, propErr = applyProperties(instance, action.properties)
        if not success then
            return false, propErr
        end
    end

    if action.property then
        local success, propErr = applyProperties(instance, {
            [action.property] = action.value
        })
        if not success then
            return false, propErr
        end
    elseif not action.properties then
        return false, "set_property action requires either 'property' and 'value' or a 'properties' table."
    end

    return true, instance
end

local function queryWorkspace(action)
    if not action or not action.query then
        return false, "query_workspace action requires a 'query' field"
    end
    
    local query = string.lower(tostring(action.query))
    -- Allow unlimited results - AI can query as many times as needed
    local maxResults = action.max_results or 999999 -- Effectively unlimited, but can be overridden if needed
    local searchType = action.search_type or "all" -- "all", "scripts", "modules", "instances"
    
    local results = {
        scripts = {},
        modules = {},
        instances = {}
    }
    
    -- Ensure scriptCache is up to date
    pcall(function()
        return scriptAnalysis.cacheAllScripts()
    end)
    
    -- Search through ALL scripts in scriptCache (no limit)
    local scriptCache = scriptAnalysis.scriptCache or {}
    for _, scriptInfo in ipairs(scriptCache) do
        local scriptName = string.lower(scriptInfo.name or "")
        local scriptPath = string.lower(scriptInfo.path or "")
        local scriptSource = string.lower(scriptInfo.source or "")
        
        -- Check if query matches name, path, or source
        local matchesName = scriptName:find(query, 1, true) ~= nil
        local matchesPath = scriptPath:find(query, 1, true) ~= nil
        local matchesSource = scriptSource:find(query, 1, true) ~= nil
        
        if matchesName or matchesPath or matchesSource then
            local isModule = scriptInfo.className == "ModuleScript"
            local resultEntry = {
                path = scriptInfo.path,
                name = scriptInfo.name,
                class = scriptInfo.className,
                source = scriptInfo.source, -- Full source, no truncation
                lineCount = scriptInfo.lineCount,
                hash = scriptInfo.sourceHash
            }
            
            if isModule then
                if searchType == "all" or searchType == "modules" then
                    table.insert(results.modules, resultEntry)
                end
            else
                if searchType == "all" or searchType == "scripts" then
                    table.insert(results.scripts, resultEntry)
                end
            end
            
            -- Only limit if explicitly requested and reasonable
            if maxResults < 999999 and (#results.scripts + #results.modules >= maxResults) then
                break
            end
        end
    end
    
    -- Search hierarchy for instances (if requested) - no limit
    if searchType == "all" or searchType == "instances" then
        -- First, search snapshot if available
        if snapshotState.current and snapshotState.current.map then
            for path, className in pairs(snapshotState.current.map) do
                if string.lower(path):find(query, 1, true) or string.lower(className):find(query, 1, true) then
                    table.insert(results.instances, {
                        path = path,
                        class = className
                    })
                    -- Only limit if explicitly requested
                    if maxResults < 999999 and #results.instances >= maxResults then
                        break
                    end
                end
            end
        end
        
        -- Also do a direct search in important services as fallback (in case snapshot is stale)
        -- This ensures we find instances even if they were added after the last snapshot
        if #results.instances == 0 then
            for _, serviceName in ipairs(importantServices) do
                local success, service = pcall(function()
                    return game:GetService(serviceName)
                end)
                if success and service then
                    -- Search through all descendants of this service
                    local descendants = service:GetDescendants()
                    for _, instance in ipairs(descendants) do
                        local instanceName = string.lower(instance.Name)
                        local fullPath = instance:GetFullName()
                        local pathLower = string.lower(fullPath)
                        local className = string.lower(instance.ClassName)
                        
                        if instanceName:find(query, 1, true) or pathLower:find(query, 1, true) or className:find(query, 1, true) then
                            -- Check if not already in results
                            local alreadyFound = false
                            for _, existing in ipairs(results.instances) do
                                if existing.path == fullPath then
                                    alreadyFound = true
                                    break
                                end
                            end
                            
                            if not alreadyFound and not instance:IsA("LuaSourceContainer") then
                                table.insert(results.instances, {
                                    path = fullPath,
                                    class = instance.ClassName
                                })
                                -- Only limit if explicitly requested
                                if maxResults < 999999 and #results.instances >= maxResults then
                                    break
                                end
                            end
                        end
                    end
                    -- Break if we've reached the limit
                    if maxResults < 999999 and #results.instances >= maxResults then
                        break
                    end
                end
            end
        end
    end
    
    -- Return results as a formatted string that can be included in assistant_message
    local resultText = {}
    table.insert(resultText, string.format("Workspace Query Results for '%s':", action.query))
    table.insert(resultText, "")
    
    if #results.modules > 0 then
        table.insert(resultText, string.format("Found %d ModuleScript(s):", #results.modules))
        for i, mod in ipairs(results.modules) do
            table.insert(resultText, string.format("%d. %s (%s) - %d lines", i, mod.path, mod.class, mod.lineCount))
            table.insert(resultText, "   Full source:")
            table.insert(resultText, "```lua")
            -- Include FULL source code - no truncation, AI needs to see everything
            table.insert(resultText, mod.source)
            table.insert(resultText, "```")
            table.insert(resultText, "")
        end
    end
    
    if #results.scripts > 0 then
        table.insert(resultText, string.format("Found %d Script(s):", #results.scripts))
        for i, script in ipairs(results.scripts) do
            table.insert(resultText, string.format("%d. %s (%s) - %d lines", i, script.path, script.class, script.lineCount))
            table.insert(resultText, "   Full source:")
            table.insert(resultText, "```lua")
            -- Include FULL source code - no truncation, AI needs to see everything
            table.insert(resultText, script.source)
            table.insert(resultText, "```")
            table.insert(resultText, "")
        end
    end
    
    if #results.instances > 0 then
        table.insert(resultText, string.format("Found %d Instance(s):", #results.instances))
        for i, inst in ipairs(results.instances) do
            table.insert(resultText, string.format("%d. %s (%s)", i, inst.path, inst.class))
        end
    end
    
    if #results.modules == 0 and #results.scripts == 0 and #results.instances == 0 then
        table.insert(resultText, "No matches found.")
    end
    
    return true, table.concat(resultText, "\n")
end

local function deleteInstance(action)
    if not action or not action.path then
        return false, "delete_instance action requires a 'path' field"
    end
    
    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = false
    })
    
    if not instance then
        return false, err or ("Instance not found at path: " .. action.path)
    end
    
    -- Don't allow deleting critical services
    if instance == game or instance.Parent == nil then
        return false, "Cannot delete root instance or service"
    end
    
    local instanceName = instance.Name
    local success, deleteErr = pcall(function()
        instance:Destroy()
    end)
    
    if not success then
        return false, "Failed to delete instance: " .. tostring(deleteErr)
    end
    
    return true, "Deleted instance: " .. instanceName
end

local function renameInstance(action)
    if not action or not action.path or not action.new_name then
        return false, "rename_instance action requires 'path' and 'new_name' fields"
    end
    
    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = false
    })
    
    if not instance then
        return false, err or ("Instance not found at path: " .. action.path)
    end
    
    local oldName = instance.Name
    local success, renameErr = pcall(function()
        instance.Name = action.new_name
    end)
    
    if not success then
        return false, "Failed to rename instance: " .. tostring(renameErr)
    end
    
    return true, string.format("Renamed '%s' to '%s'", oldName, action.new_name)
end

local function moveInstance(action)
    if not action or not action.path or not action.new_parent then
        return false, "move_instance action requires 'path' and 'new_parent' fields"
    end
    
    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = false
    })
    
    if not instance then
        return false, err or ("Instance not found at path: " .. action.path)
    end
    
    local newParent, parentErr = resolveInstanceFromPath(action.new_parent, {
        createMissing = action.create_parent_if_missing or false,
        intermediateClass = "Folder"
    })
    
    if not newParent then
        return false, parentErr or ("New parent not found at path: " .. action.new_parent)
    end
    
    local instanceName = instance.Name
    local success, moveErr = pcall(function()
        instance.Parent = newParent
    end)
    
    if not success then
        return false, "Failed to move instance: " .. tostring(moveErr)
    end
    
    return true, string.format("Moved '%s' to '%s'", instanceName, action.new_parent)
end

local function cloneInstance(action)
    if not action or not action.path then
        return false, "clone_instance action requires a 'path' field"
    end
    
    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = false
    })
    
    if not instance then
        return false, err or ("Instance not found at path: " .. action.path)
    end
    
    local newName = action.new_name or (instance.Name .. " (Copy)")
    local newParent = instance.Parent
    
    if action.new_parent then
        local resolvedParent, parentErr = resolveInstanceFromPath(action.new_parent, {
            createMissing = action.create_parent_if_missing or false,
            intermediateClass = "Folder"
        })
        if resolvedParent then
            newParent = resolvedParent
        end
    end
    
    local success, cloneErr = pcall(function()
        local clone = instance:Clone()
        clone.Name = newName
        clone.Parent = newParent
        return clone
    end)
    
    if not success then
        return false, "Failed to clone instance: " .. tostring(cloneErr)
    end
    
    return true, string.format("Cloned '%s' as '%s'", instance.Name, newName)
end

local function setTags(action)
    if not action or not action.path then
        return false, "set_tags action requires a 'path' field"
    end
    
    if not CollectionService then
        return false, "CollectionService is not available"
    end
    
    local instance, err = resolveInstanceFromPath(action.path, {
        createMissing = false
    })
    
    if not instance then
        return false, err or ("Instance not found at path: " .. action.path)
    end
    
    local tagsToAdd = action.add_tags or {}
    local tagsToRemove = action.remove_tags or {}
    
    local added = {}
    local removed = {}
    
    -- Add tags
    for _, tag in ipairs(tagsToAdd) do
        if type(tag) == "string" and tag ~= "" then
            local success = pcall(function()
                CollectionService:AddTag(instance, tag)
            end)
            if success then
                table.insert(added, tag)
            end
        end
    end
    
    -- Remove tags
    for _, tag in ipairs(tagsToRemove) do
        if type(tag) == "string" and tag ~= "" then
            local success = pcall(function()
                CollectionService:RemoveTag(instance, tag)
            end)
            if success then
                table.insert(removed, tag)
            end
        end
    end
    
    local resultParts = {}
    if #added > 0 then
        table.insert(resultParts, "Added tags: " .. table.concat(added, ", "))
    end
    if #removed > 0 then
        table.insert(resultParts, "Removed tags: " .. table.concat(removed, ", "))
    end
    
    if #resultParts == 0 then
        return false, "No tags were added or removed"
    end
    
    return true, table.concat(resultParts, ". ")
end

local function executeAgentAction(action)
    if not action or not action.type then
        return false, "Invalid agent action payload"
    end

    if action.type == "query_workspace" then
        return queryWorkspace(action)
    elseif action.type == "ensure_instance" then
        return ensureInstance(action)
    elseif action.type == "write_script" then
        return writeScriptSource(action)
    elseif action.type == "set_property" then
        return setProperty(action)
    elseif action.type == "delete_instance" then
        return deleteInstance(action)
    elseif action.type == "rename_instance" then
        return renameInstance(action)
    elseif action.type == "move_instance" then
        return moveInstance(action)
    elseif action.type == "clone_instance" then
        return cloneInstance(action)
    elseif action.type == "set_tags" then
        return setTags(action)
    else
        return false, "Unsupported action type: " .. tostring(action.type)
    end
end

local function executeAgentActions(actions)
    if not actions or #actions == 0 then
        return true
    end

    -- Separate query_workspace actions from other actions - execute queries first
    local queryActions = {}
    local otherActions = {}
    
    for _, action in ipairs(actions) do
        if action.type == "query_workspace" then
            table.insert(queryActions, action)
        else
            table.insert(otherActions, action)
        end
    end
    
    -- Execute query actions first and collect results
    local queryResults = {}
    local willTriggerContinuation = false -- Track if we'll trigger continuation after queries
        if #queryActions > 0 then
        -- Check if we'll need to trigger continuation BEFORE executing queries
        -- This ensures the flag is set before runAgentActions callback completes
        -- Allow continuation if: no other actions AND (haven't done continuation OR it's an allowed additional query cycle)
        local canContinuation = #otherActions == 0 and (
            not errorState.queryContinuationDone or 
            (errorState.queryOnlyContinuationCount or 0) < 2
        )
        if canContinuation then
            willTriggerContinuation = true
            -- CRITICAL: Set flag BEFORE executing queries so callback sees it
            errorState.queryContinuationPending = true
            -- Only set queryContinuationDone on first query cycle, allow subsequent cycles
            if not errorState.queryContinuationDone then
                errorState.queryContinuationDone = true
            end
        end
        
        -- Execute all queries in parallel for faster processing
        local queryThreads = {}
        local queryResultsArray = {}
        local completedCount = 0
        local totalQueries = #queryActions
        
        -- Spawn all query tasks in parallel
        for i, action in ipairs(queryActions) do
            task.spawn(function()
                local ok, res = executeAgentAction(action)
                if ok then
                    queryResultsArray[i] = res
                else
                    warn("Query action failed:", res)
                    queryResultsArray[i] = nil
                end
                completedCount = completedCount + 1
            end)
        end
        
        -- Wait for all queries to complete (with timeout safety)
        local timeout = 0
        while completedCount < totalQueries and timeout < 300 do
            task.wait(0.1)
            timeout = timeout + 1
        end
        
        -- Collect results in order
        for i = 1, totalQueries do
            if queryResultsArray[i] then
                table.insert(queryResults, queryResultsArray[i])
            end
        end
        
        -- Include query results in conversation so AI can see them
        if #queryResults > 0 then
            local combinedResults = table.concat(queryResults, "\n\n")
            appendConversationMessage(
                "system",
                "Workspace Query Results:\n" .. combinedResults .. "\n\n[Continue implementing the original user request using the query results above.]",
                nil,
                true
            )
            
            -- If there are no other actions, automatically continue the same request seamlessly
            -- Query results are now in conversation context, so AI can generate new actions based on them
            -- For continuation queries (when queryOnlyContinuationCount > 0), also allow continuation
            -- Check if we should trigger continuation - either from initial check or if this is a follow-up query cycle
            if not willTriggerContinuation and errorState.queryContinuationPending and (errorState.queryOnlyContinuationCount or 0) > 0 then
                -- This is queries from a continuation response - check if we can do another continuation
                local canAnotherContinuation = (errorState.queryOnlyContinuationCount or 0) < 2
                if canAnotherContinuation then
                    willTriggerContinuation = true
                end
            end
            
            if willTriggerContinuation then
                    -- Send a seamless continuation request - query results are in conversation context
                    -- Extract the original user message to reference in the continuation message
                    local originalUserMessage = nil
                    for i = #agentState.conversation, 1, -1 do
                        if agentState.conversation[i].role == "user" then
                            originalUserMessage = agentState.conversation[i].content
                            break
                        end
                    end
                    
                    if originalUserMessage then
                        local apiKey = getActiveApiKey()
                        if apiKey ~= "" then
                            saveAgentSession()
                            -- Send seamless continuation - explicitly tell AI to analyze query results and implement
                            -- Backend will see both original request and query results, generating actions accordingly
                            -- Note: queryContinuationPending and queryContinuationDone were already set above before queries executed
                            -- This ensures the callback sees the flag and doesn't finalize prematurely
                            -- Use a small delay to ensure queries are fully processed before continuation
                            -- Don't reset UI state - keep it looking like the same request is continuing
                            task.delay(0.5, function()
                                -- Keep request in flight (already true from original request)
                                -- Don't reset agentRequestInFlight - it should already be true
                                agentState.pendingAssistantChunks = 0
                                -- Keep SendButton in "Thinking..." state - don't reset it
                                -- Keep thought stages visible - don't hide/show them unnecessarily
                                -- Only restart narration if it stopped, otherwise keep it going seamlessly
                                if not agentState.liveNarrationActive then
                                    startLiveNarration()
                                end
                                -- Keep thought stages visible - don't reset them for continuations
                                -- Don't hide/show - keep continuous UI state
                                -- For query continuations, keep thought stage at current level (don't reset to 1)
                                -- This prevents visual "reopening" - the UI should stay continuous
                                -- Don't call showThoughtStages() - keep it looking like the same request
                                
                                local function performRequest(snapshot)
                                    -- Create a continuation message that explicitly tells AI to analyze query results
                                    -- The system message already contains the query results, so this message should
                                    -- instruct the AI to analyze them and then implement the original request
                                    local queryCycleNote = ""
                                    if errorState.queryOnlyContinuationCount and errorState.queryOnlyContinuationCount > 0 then
                                        queryCycleNote = "I've completed additional workspace queries. "
                                    else
                                        queryCycleNote = "I've queried the workspace and retrieved the results above. "
                                    end
                                    local continuationMessage = string.format(
                                        "%sPlease analyze these query results, understand the current codebase structure, and then implement the original request: \"%s\". If you need additional information, you can query again, but please proceed with implementation after reviewing the available query results.",
                                        queryCycleNote,
                                        originalUserMessage
                                    )
                                    
                                    print("[Electrode] Continuation request: Parsing prompt and preparing request for backend...")
                                    print("[Electrode] Continuation prompt:", continuationMessage)
                                    
                                    local payload = {
                                        api_key = apiKey,
                                        model = agentState.model,
                                        trusted = agentState.trusted,
                                        session_id = agentState.sessionId, -- Same session
                                        conversation = serializeConversation(), -- Includes query results as system message
                                        snapshot = buildSnapshotPayload(snapshot),
                                        plan = agentState.plan,
                                        roblox_user_id = Network.getRobloxUserId(),
                                        request = {
                                            message = continuationMessage, -- Explicit instruction to analyze query results and implement
                                            timestamp = os.time()
                                        }
                                    }
                                    
                                    print("[Electrode] Continuation payload prepared, sending to backend...")
                                    print("[Electrode] Backend will retrieve relevant context blocks based on prompt tags")
                                    
                                    local ok, response, httpError = pcall(function()
                                        -- Don't reset thought stage - keep it continuous
                                        -- For query continuations, we want to stay at stage 3 (analyzing) or higher
                                        -- The thought stages should already be visible from the original request
                                        -- Only update to stage 3 if we need to indicate we're analyzing
                                        if AgentUI.ThoughtContainer and AgentUI.ThoughtContainer.Visible then
                                            setThoughtStage(3) -- Analyzing query results and generating implementation
                                        end
                                        return Network.requestAgentResponse(apiKey, payload)
                                    end)
                                    
                                    -- Don't reset queryContinuationPending yet - wait until we see the response
                                    -- This ensures the request stays open during query → continuation → implementation flow
                                    if not ok then
                                        -- Continuation request failed - reset flags and finalize
                                        errorState.queryContinuationPending = false
                                        errorState.queryContinuationDone = false
                                        errorState.queryOnlyContinuationCount = 0
                                        handleResponse(nil, response, true) -- Pass isContinuation=true so continuation logic applies
                                    else
                                        -- Response will be handled by handleResponse, which will reset queryContinuationPending
                                        -- when implementation actions complete
                                        -- CRITICAL: Pass isContinuation=true so the response is treated as a continuation
                                        -- This prevents premature finalization if continuation returns only queries
                                        handleResponse(response, httpError, true)
                                    end
                                end
                                
                                local function proceedWithSnapshot(snapshot)
                                    -- Don't reset thought stage for query continuations - keep it continuous
                                    performRequest(snapshot or snapshotState.current)
                                end
                                
                                if snapshotState.capturing then
                                    -- Don't reset thought stage - keep UI continuous
                                    captureSnapshot("agent-continuation", proceedWithSnapshot)
                                elseif snapshotState.current and not snapshotState.dirty then
                                    -- Don't reset thought stage - keep UI continuous
                                    proceedWithSnapshot(snapshotState.current)
                                else
                                    -- Don't reset thought stage - keep UI continuous
                                    captureSnapshot("agent-continuation", proceedWithSnapshot)
                                end
                            end)
                            -- Return true to indicate query was successful, continuation will handle the rest
                            return true
                        end
                    end
            end
        end
    end
    
    -- If we triggered continuation and there are no other actions, return true immediately
    -- BUT: Don't let the callback finalize - the continuation will handle everything
    -- We need to return true so runAgentActions knows queries succeeded,
    -- but the callback will check queryContinuationPending and return early
    if willTriggerContinuation and #otherActions == 0 then
        -- Continuation is pending - flag is already set, callback will see it and return early
        -- Return true to indicate queries succeeded
        return true
    end
    
    -- Now execute other actions
    local waypointName = "Electrode Agent Changes"
    ChangeHistoryService:SetWaypoint(waypointName .. " (Before)")

    local totalActions = #otherActions
    local failedActions = {}
    local successfulActions = 0
    
    for index, action in ipairs(otherActions) do
        local headline = describeActionHeadline(action)
        if headline then
            appendConversationMessage(
                "assistant",
                string.format("Applying step %d/%d: %s...", index, totalActions, headline),
                nil,
                true
            )
        end
        
        local success, result = executeAgentAction(action)
        if not success then
            warn("Agent action failed:", result)
            table.insert(failedActions, {
                index = index,
                action = action,
                error = tostring(result),
                headline = headline
            })
            
            -- Try to provide helpful error message with actionable feedback
            local errorMsg = tostring(result)
            local isMissingInstance = string.find(errorMsg, "Missing instance", 1, true)
            
            if string.find(errorMsg, "Unknown service", 1, true) then
                appendConversationMessage(
                    "assistant",
                    string.format("⚠️ Step %d failed: %s. I'll continue with the other steps.", index, errorMsg),
                    nil,
                    true
                )
            elseif isMissingInstance then
                -- For missing instance errors, suggest creating it first
                appendConversationMessage(
                    "assistant",
                    string.format("⚠️ Step %d failed: %s. The instance doesn't exist - I may need to create it first using ensure_instance before modifying it. Continuing with remaining steps...", index, errorMsg),
                    nil,
                    true
                )
            else
                appendConversationMessage(
                    "assistant",
                    string.format("⚠️ Step %d had an issue: %s. Continuing with remaining steps...", index, errorMsg),
                    nil,
                    true
                )
            end
        else
            successfulActions = successfulActions + 1
        end
    end

    ChangeHistoryService:SetWaypoint(waypointName .. " (After)")
    
    -- Report results
    if #failedActions > 0 then
        local successRate = math.floor((successfulActions / totalActions) * 100)
        if successfulActions > 0 then
            appendConversationMessage(
                "assistant",
                string.format("✅ Completed %d/%d steps (%d%%). %d step(s) had issues but I continued with the rest.", 
                    successfulActions, totalActions, successRate, #failedActions),
                nil,
                true
            )
            -- Status messages shown in chat, no Generator status label needed
        else
            appendConversationMessage(
                "assistant",
                string.format("❌ All %d steps failed. Check the errors above for details.", totalActions),
                nil,
                true
            )
            return false, "All actions failed"
        end
    else
        appendConversationMessage(
            "assistant",
            string.format("✅ All %d steps completed successfully!", totalActions),
            nil,
            true
        )
    end
    
    return true
end

local function summarizeAgentActions(actions)
    local lines = {}
    local total = #(actions or {})
    for index, action in ipairs(actions or {}) do
        if index > 3 then
            table.insert(lines, string.format("• ...and %d more action(s)", total - 3))
            break
        end
        if action.type == "query_workspace" then
            table.insert(lines, string.format("• Search workspace for '%s'", action.query or "?"))
        elseif action.type == "write_script" then
            table.insert(lines, string.format("• Write %s at %s", action.class or "Script", action.path or "?"))
        elseif action.type == "ensure_instance" then
            table.insert(lines, string.format("• Ensure %s exists at %s", action.class or "Folder", action.path or "?"))
        elseif action.type == "set_property" then
            table.insert(lines, string.format("• Set %s.%s = %s", action.path or "?", action.property or "Property", tostring(action.value)))
        elseif action.type == "delete_instance" then
            table.insert(lines, string.format("• Delete %s", action.path or "?"))
        elseif action.type == "rename_instance" then
            table.insert(lines, string.format("• Rename %s to '%s'", action.path or "?", action.new_name or "?"))
        elseif action.type == "move_instance" then
            table.insert(lines, string.format("• Move %s to %s", action.path or "?", action.new_parent or "?"))
        elseif action.type == "clone_instance" then
            table.insert(lines, string.format("• Clone %s", action.path or "?"))
        elseif action.type == "set_tags" then
            table.insert(lines, string.format("• Update tags on %s", action.path or "?"))
        else
            table.insert(lines, string.format("• %s", action.type or "Unknown action"))
        end
    end
    return table.concat(lines, "\n")
end

function runAgentActions(actions, onComplete)
    if not actions or #actions == 0 then
        if onComplete then
            onComplete(true)
        end
        return
    end

    local function finalize(success, err)
        if success then
            captureSnapshot("agent-actions")
        end
        if onComplete then
            onComplete(success, err)
        end
    end

    if agentState.trusted then
        local success, err = executeAgentActions(actions)
        finalize(success, err)
    else
        local summary = summarizeAgentActions(actions)
        local message = "Electrode wants to apply these changes:\n\n" .. summary .. "\n\nApply changes?"
        showConfirmationDialog(message, "Apply Changes", "Cancel", function()
            local success, err = executeAgentActions(actions)
            finalize(success, err)
        end, function()
            finalize(false, "User cancelled")
        end)
    end
end

-- Generator placement buttons removed (AutoPlaceButton, ManualPlaceButton)

AgentUI.SendButton.MouseButton1Click:Connect(sendAgentMessage)

local function setRefreshButtonIdle()
    AgentUI.RefreshContextButton.Text = "Refresh Snapshot"
    Style.setButtonGradient(
        AgentUI.RefreshContextButton,
        Color3.fromRGB(175, 145, 255),
        Color3.fromRGB(96, 128, 255),
        NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.05),
            NumberSequenceKeypoint.new(1, 0.2)
        })
    )
    local stroke = AgentUI.RefreshContextButton:FindFirstChild("ButtonStroke")
    if stroke then
        stroke.Color = Color3.fromRGB(190, 170, 255)
        stroke.Transparency = 0.15
    end
    AgentUI.RefreshContextButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    AgentUI.RefreshContextButton.Active = true
end

setRefreshButtonIdle()

AgentUI.RefreshContextButton.MouseButton1Click:Connect(function()
    if snapshotState.capturing then
        return
    end

    AgentUI.RefreshContextButton.Text = "Refreshing..."
    Style.setButtonGradient(
        AgentUI.RefreshContextButton,
        Color3.fromRGB(255, 210, 120),
        Color3.fromRGB(255, 170, 50),
        NumberSequence.new({
            NumberSequenceKeypoint.new(0, 0.05),
            NumberSequenceKeypoint.new(1, 0.25)
        })
    )
    local stroke = AgentUI.RefreshContextButton:FindFirstChild("ButtonStroke")
    if stroke then
        stroke.Color = Color3.fromRGB(255, 214, 140)
        stroke.Transparency = 0.2
    end
    AgentUI.RefreshContextButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    AgentUI.RefreshContextButton.Active = false

    captureSnapshot("manual", function(_, _, err)
        if err then
            AgentUI.RefreshContextButton.Text = "Retry Snapshot"
            Style.setButtonGradient(
                AgentUI.RefreshContextButton,
                Color3.fromRGB(255, 120, 120),
                Color3.fromRGB(220, 60, 80),
                NumberSequence.new({
                    NumberSequenceKeypoint.new(0, 0.05),
                    NumberSequenceKeypoint.new(1, 0.25)
                })
            )
            local failureStroke = AgentUI.RefreshContextButton:FindFirstChild("ButtonStroke")
            if failureStroke then
                failureStroke.Color = Color3.fromRGB(255, 160, 160)
                failureStroke.Transparency = 0.2
            end
            AgentUI.RefreshContextButton.TextColor3 = Color3.fromRGB(255, 255, 255)
            AgentUI.RefreshContextButton.Active = true
            AgentUI.SnapshotTimestampLabel.Text = "Snapshot failed"
            AgentUI.SnapshotSummaryLabel.Text = typeof(err) == "string" and err or "Check Output for details."
            return
        end

        setRefreshButtonIdle()
    end)
end)

AgentUI.ChatInputBox.FocusLost:Connect(function(enterPressed)
    if enterPressed then
        sendAgentMessage()
    end
end)

-- Game context storage
-- Attach to snapshotState to avoid module-level locals (access directly, no local aliases)
snapshotState.gameContext = nil
snapshotState.contextSyncTime = nil

-- Helper: Build tree view of hierarchy (attach to scriptAnalysis)
scriptAnalysis.buildTreeView = function(parent, depth, maxDepth, prefix)
    if depth >= maxDepth then return "" end
    
    local lines = {}
    local children = parent:GetChildren()
    
    for i, child in ipairs(children) do
        local isLast = (i == #children)
        local connector = isLast and "└── " or "├── "
        local newPrefix = isLast and "    " or "│   "
        
        local line = prefix .. connector .. child.Name
        
        -- Add type info for non-folders
        if not child:IsA("Folder") and not child:IsA("Model") then
            line = line .. " (" .. child.ClassName .. ")"
        end
        
        table.insert(lines, line)
        
        -- Recurse for containers
        if child:IsA("Folder") or child:IsA("Model") and depth < maxDepth - 1 then
            local subtree = scriptAnalysis.buildTreeView(child, depth + 1, maxDepth, prefix .. newPrefix)
            if subtree ~= "" then
                table.insert(lines, subtree)
            end
        end
    end
    
    return table.concat(lines, "\n")
end

-- Helper: Find all RemoteEvents and RemoteFunctions (attach to scriptAnalysis)
scriptAnalysis.findAllRemotes = function()
    local remotes = {}
    
    for _, obj in ipairs(game:GetDescendants()) do
        if obj:IsA("RemoteEvent") or obj:IsA("RemoteFunction") then
            table.insert(remotes, {
                path = obj:GetFullName(),
                type = obj.ClassName
            })
        end
    end
    
    return remotes
end
-- Removed unnecessary wrapper function to reduce register usage

-- Wrap script analysis functions in do...end to reduce module-level locals
-- This creates a separate scope so these functions don't count toward the 200-local limit
-- Use a single table to hold all functions (1 local instead of 8)
-- scriptAnalysis already initialized at top of file
do
    scriptAnalysis.simpleHash = function(str)
        if not str or str == "" then
            return "0"
        end
        local hash = 0
        for i = 1, #str do
            hash = (hash * 33 + str:byte(i)) % 4294967291
        end
        return string.format("%x", hash)
    end

    -- Script cache for agent-like searching
    scriptAnalysis.scriptCache = {}

    -- Deep script analysis: Extract structure, dependencies, and patterns
    -- METHOD 1: Deep Script Parsing and Analysis
    scriptAnalysis.analyzeScriptStructure = function(source, path, className)
        local analysis = {
            functions = {},
            exports = {},
            requires = {},
            variables = {},
            services = {},
            patterns = {}
        }
        
        if not source or #source < 10 then
            return analysis
        end
        
        -- Helper to normalize parameters (reduces duplicate code and locals)
        local function normalizeParams(p) return p:gsub("%s+", " "):gsub("^%s+", ""):gsub("%s+$", "") end
        
        -- Extract function signatures (handles both "function name()" and "name = function()")
        for funcName, params in source:gmatch("function%s+([%w_%.]+)%s*%(([^)]*)%)") do
            local cleanName = funcName:match("([%w_]+)$") or funcName
            local normParams = normalizeParams(params)
            table.insert(analysis.functions, {name = cleanName, parameters = normParams, signature = cleanName .. "(" .. normParams .. ")"})
        end
        
        -- Extract function assignments (name = function(params))
        for funcName, params in source:gmatch("([%w_]+)%s*=%s*function%s*%(([^)]*)%)") do
            local normParams = normalizeParams(params)
            table.insert(analysis.functions, {name = funcName, parameters = normParams, signature = funcName .. "(" .. normParams .. ")"})
        end
        
        -- Extract module exports (for ModuleScripts) - optimized
        if className == "ModuleScript" then
            for returnBlock in source:gmatch("return%s+({[^}]+})") do
                for key in returnBlock:gmatch("([%w_]+)%s*=") do
                    analysis.exports[key] = true
                end
            end
            if source:find("return%s+[%w_]+%s*$") or source:find("return%s+[%w_]+%s*;") then
                for moduleName in source:gmatch("local%s+([%w_]+)%s*=%s*{}") do
                    for key in source:gmatch(moduleName .. "%[?[\"']?([%w_]+)[\"']?%]?%s*=") do
                        analysis.exports[key] = true
                    end
                end
            end
        end
        
        -- Extract require statements (handles various formats) - optimized to reduce locals
        for requireCall in source:gmatch("require%([^)]+%)") do
            local requirePath = requireCall:match("require%(([^)]+)%)")
            if requirePath and not requirePath:find("game:GetService") then
                requirePath = requirePath:gsub("[\"']", ""):gsub("^%s+", ""):gsub("%s+$", "")
                requirePath = requirePath:gsub("^game%.", ""):gsub("%[\"']([%w_]+)[\"']%]", "%1"):gsub("%.", "/")
                if requirePath and #requirePath > 0 and not requirePath:match("^%s*$") then
                    analysis.requires[requirePath] = true
                end
            end
        end
        
        -- Extract variable declarations (local varName = ...)
        for varName in source:gmatch("local%s+([%w_]+)%s*=") do
            if #varName > 2 and varName ~= "function" then
                analysis.variables[varName] = true
            end
        end
        
        -- Extract service usage patterns
        for serviceName in source:gmatch("game:GetService%(\"([%w]+)\"%)") do
            analysis.services[serviceName] = true
        end
        
        -- Detect code patterns (single-line to reduce locals)
        if source:find("pcall") then analysis.patterns.hasErrorHandling = true end
        if source:find(":Connect") or source:find(":connect()") then analysis.patterns.hasEventHandling = true end
        if source:find("task%.spawn") or source:find("spawn%(") then analysis.patterns.hasAsyncExecution = true end
        if source:find("task%.wait") or source:find("wait%(") then analysis.patterns.hasYielding = true end
        if source:find(":GetAttribute") or source:find(":SetAttribute") then analysis.patterns.usesAttributes = true end
        if source:find("CollectionService") then analysis.patterns.usesCollectionService = true end
        
        -- Convert tables to arrays for JSON serialization (reuse same variable to reduce locals)
        local result = {functions = analysis.functions, patterns = analysis.patterns}
        result.exports = {}
        for key in pairs(analysis.exports) do table.insert(result.exports, key) end
        result.requires = {}
        for path in pairs(analysis.requires) do table.insert(result.requires, path) end
        result.variables = {}
        for var in pairs(analysis.variables) do table.insert(result.variables, var) end
        result.services = {}
        for service in pairs(analysis.services) do table.insert(result.services, service) end
        return result
    end

    -- Build dependency graph from script cache
    -- Maps script paths to their dependencies and exports
    scriptAnalysis.buildDependencyGraph = function()
        local graph = {}
        
        -- Helper: Resolve require path to actual script path
        local function resolveRequirePath(requirePath, currentScriptPath)
            -- Try direct path match
            for _, scriptInfo in ipairs(scriptAnalysis.scriptCache) do
                local scriptPath = scriptInfo.path:gsub("%.", "/")
                if scriptPath:find(requirePath, 1, true) or requirePath:find(scriptPath, 1, true) then
                    return scriptInfo.path
                end
                
                -- Try name match
                if scriptInfo.name == requirePath or scriptInfo.name:find(requirePath, 1, true) then
                    return scriptInfo.path
                end
            end
            
            -- Try partial path matching
            local pathParts = {}
            for part in requirePath:gmatch("[^/]+") do
                table.insert(pathParts, part)
            end
            
            if #pathParts > 0 then
                local lastName = pathParts[#pathParts]
                for _, scriptInfo in ipairs(scriptAnalysis.scriptCache) do
                    if scriptInfo.name == lastName then
                        return scriptInfo.path
                    end
                end
            end
            
            return nil
        end
        
        -- Build graph for each script
        for _, scriptInfo in ipairs(scriptAnalysis.scriptCache) do
            local analysis = scriptInfo.analysis
            if analysis then
                local resolvedRequires = {}
                for _, requirePath in ipairs(analysis.requires) do
                    local resolved = resolveRequirePath(requirePath, scriptInfo.path)
                    if resolved then
                        table.insert(resolvedRequires, resolved)
                    end
                end
                
                graph[scriptInfo.path] = {
                    requires = analysis.requires,
                    resolvedRequires = resolvedRequires,
                    exports = analysis.exports,
                    functions = analysis.functions,
                    services = analysis.services,
                    patterns = analysis.patterns
                }
            end
        end
        
        return graph
    end

    -- Cache all scripts in the game for intelligent searching
    -- No limit - cache ALL scripts so AI can read through everything it needs
    -- Now includes deep analysis of script structure, dependencies, and patterns
    scriptAnalysis.cacheAllScripts = function()
        scriptAnalysis.scriptCache = {}
        local cacheCount = 0
        
        for _, obj in ipairs(game:GetDescendants()) do
            if obj:IsA("Script") or obj:IsA("LocalScript") or obj:IsA("ModuleScript") then
                local success, source = pcall(function()
                    return obj.Source
                end)
                
                if success and source and #source > 10 then
                    local _, newlineCount = source:gsub("\n", "")
                    local lineCount = newlineCount + 1
                    local path = obj:GetFullName()
                    
                    -- Perform deep analysis of script structure
                    local analysis = scriptAnalysis.analyzeScriptStructure(source, path, obj.ClassName)
                    
                    table.insert(scriptAnalysis.scriptCache, {
                        name = obj.Name,
                        path = path,
                        source = source, -- Full source, no truncation
                        parent = obj.Parent and obj.Parent.Name or "Unknown",
                        className = obj.ClassName,
                        lineCount = lineCount,
                        sourceHash = scriptAnalysis.simpleHash(source),
                        analysis = analysis -- Deep analysis: functions, exports, requires, patterns
                    })
                    cacheCount = cacheCount + 1
                end
            end
        end
        
        return cacheCount
    end

    -- Extract keywords from user prompt for intelligent script searching
    scriptAnalysis.extractKeywords = function(prompt)
        local keywords = {}
        local relatedTerms = {}
        
        local commonWords = {
        ["the"] = true, ["a"] = true, ["an"] = true, ["and"] = true, ["or"] = true,
        ["but"] = true, ["in"] = true, ["on"] = true, ["at"] = true, ["to"] = true,
        ["for"] = true, ["of"] = true, ["with"] = true, ["by"] = true, ["from"] = true,
        ["this"] = true, ["that"] = true, ["it"] = true, ["is"] = true, ["are"] = true,
        ["can"] = true, ["make"] = true, ["create"] = true, ["add"] = true, ["write"] = true,
        ["script"] = true, ["code"] = true, ["please"] = true, ["want"] = true, ["need"] = true,
        ["build"] = true, ["system"] = true
    }
    
    -- Related systems mapping: if asking for X, also look for Y
    local relatedSystems = {
        shop = {"inventory", "currency", "purchase", "coins", "money", "player", "data", "gui"},
        weapon = {"damage", "tool", "combat", "player", "inventory", "equip"},
        inventory = {"player", "data", "item", "equip", "storage"},
        damage = {"health", "combat", "player", "weapon", "attack"},
        teleport = {"player", "position", "spawn", "checkpoint"},
        quest = {"player", "data", "objective", "reward", "progress"},
        npc = {"dialog", "interaction", "player", "character"},
        gui = {"player", "menu", "interface", "button", "frame"},
        datastore = {"player", "data", "save", "load", "persistent"},
        remote = {"server", "client", "network", "event", "function"}
    }
    
    -- Extract words longer than 3 chars that aren't common
    for word in prompt:lower():gmatch("%w+") do
        if #word > 3 and not commonWords[word] then
            keywords[word] = true
            
            -- Add related terms for better context
            if relatedSystems[word] then
                for _, related in ipairs(relatedSystems[word]) do
                    relatedTerms[related] = true
                end
            end
        end
    end
    
    return keywords, relatedTerms
end

    -- Search for relevant scripts based on user prompt (Agent-like behavior)
    -- Looks for RELATED systems, not exact matches (e.g., if asking for shop, finds inventory/GUI/data scripts)
    scriptAnalysis.findRelevantScripts = function(prompt, maxScripts)
        maxScripts = maxScripts or 5
        local keywords, relatedTerms = scriptAnalysis.extractKeywords(prompt)
        local scoredScripts = {}
        
        -- Score each cached script by relevance
        for _, script in ipairs(scriptAnalysis.scriptCache) do
            local score = 0
            local nameAndPath = (script.name .. " " .. script.path .. " " .. script.parent):lower()
            local sourceLower = script.source:lower()
            
            -- Check primary keywords in script name/path (medium-high weight)
            for keyword in pairs(keywords) do
                if nameAndPath:find(keyword, 1, true) then
                    score = score + 2  -- Reduced from 3 - might be asking to CREATE this
                end
                if sourceLower:find(keyword, 1, true) then
                    score = score + 0.5  -- Reduced - might be asking to create
                end
            end
            
            -- Check RELATED terms (higher weight - these are reference examples)
            for relatedTerm in pairs(relatedTerms) do
                if nameAndPath:find(relatedTerm, 1, true) then
                    score = score + 4  -- Higher weight - these are good reference examples!
                end
                if sourceLower:find(relatedTerm, 1, true) then
                    score = score + 2  -- Good reference code
                end
            end
            
            if score > 0 then
                table.insert(scoredScripts, {script = script, score = score})
            end
        end
        
        -- Sort by relevance score
        table.sort(scoredScripts, function(a, b) return a.score > b.score end)
        
        -- Return top N relevant scripts
        local relevant = {}
        for i = 1, math.min(maxScripts, #scoredScripts) do
            table.insert(relevant, scoredScripts[i].script)
        end
        
        return relevant
    end

    -- Analyze patterns from relevant scripts (now uses agent-like search)
    scriptAnalysis.analyzeCodePatterns = function(userPrompt)
        local p = {functions = {}, variables = {}, services = {}, relevantCode = {}}
        local s = {funcs = {}, vars = {}}
        local scripts = scriptAnalysis.findRelevantScripts(userPrompt or "", 5)
        if #scripts == 0 then
            for i = 1, math.min(5, #scriptAnalysis.scriptCache) do scripts[i] = scriptAnalysis.scriptCache[i] end
        end
        for i = 1, #scripts do
            local sc = scripts[i]
            local a = sc.analysis or scriptAnalysis.analyzeScriptStructure(sc.source, sc.path, sc.className)
            do
                local f = a.functions
                for j = 1, #f do
                    local n = f[j].name
                    if not s.funcs[n] then s.funcs[n] = true p.functions[#p.functions + 1] = n end
                end
            end
            do
                local v = a.variables
                for j = 1, #v do
                    local n = v[j]
                    if not s.vars[n] and #n > 3 then s.vars[n] = true p.variables[#p.variables + 1] = n end
                end
            end
            p.relevantCode[#p.relevantCode + 1] = {name = sc.name, path = sc.path, snippet = sc.source:sub(1, math.min(500, #sc.source))}
        end
        return p
    end
end

-- Main: Build comprehensive game context (agent-like with prompt awareness)
-- Attach to snapshotState to avoid module-level local
snapshotState.buildGameContext = function(self, userPrompt)
    local sections = {}
    
    table.insert(sections, "# GAME CONTEXT FOR AI CODE GENERATION")
    table.insert(sections, "")
    
    -- 1. Workspace Hierarchy
    table.insert(sections, "## WORKSPACE HIERARCHY")
    table.insert(sections, "```")
    table.insert(sections, "Workspace/")
    table.insert(sections, scriptAnalysis.buildTreeView(game.Workspace, 0, 3, ""))
    table.insert(sections, "```")
    table.insert(sections, "")
    
    -- 2. ReplicatedStorage
    table.insert(sections, "## REPLICATEDSTORAGE")
    table.insert(sections, "```")
    table.insert(sections, "ReplicatedStorage/")
    table.insert(sections, scriptAnalysis.buildTreeView(game.ReplicatedStorage, 0, 3, ""))
    table.insert(sections, "```")
    table.insert(sections, "")
    
    -- 3. ServerStorage (if accessible)
    local hasServerStorage = pcall(function()
        return game.ServerStorage:GetChildren()
    end)
    
    if hasServerStorage then
        table.insert(sections, "## SERVERSTORAGE")
        table.insert(sections, "```")
        table.insert(sections, "ServerStorage/")
        table.insert(sections, scriptAnalysis.buildTreeView(game.ServerStorage, 0, 2, ""))
        table.insert(sections, "```")
        table.insert(sections, "")
    end
    
    -- 4. Remote Events
    local remotes = scriptAnalysis.findAllRemotes()
    if #remotes > 0 then
        table.insert(sections, "## REMOTE EVENTS & FUNCTIONS")
        for _, remote in ipairs(remotes) do
            table.insert(sections, string.format("- `%s` (%s)", remote.path, remote.type))
        end
        table.insert(sections, "")
    end
    
    -- 5. AGENT-LIKE: Analyze relevant code based on user prompt
    local patterns = scriptAnalysis.analyzeCodePatterns(userPrompt)
    
    -- Show relevant code snippets first (most important)
    if #patterns.relevantCode > 0 then
        table.insert(sections, "## RELATED CODE EXAMPLES (For Reference)")
        table.insert(sections, "The agent found these existing scripts related to your request.")
        table.insert(sections, "Use them as reference for coding style, patterns, and structure:")
        table.insert(sections, "")
        for _, codeSnippet in ipairs(patterns.relevantCode) do
            table.insert(sections, string.format("### From `%s` (%s)", codeSnippet.name, codeSnippet.path))
            table.insert(sections, "```lua")
            table.insert(sections, codeSnippet.snippet)
            table.insert(sections, "```")
            table.insert(sections, "")
        end
    end
    
    -- Then show patterns
    if #patterns.functions > 0 or #patterns.variables > 0 then
        table.insert(sections, "## YOUR CODE PATTERNS")
        
        if #patterns.functions > 0 then
            table.insert(sections, "### Common Functions:")
            for i = 1, math.min(10, #patterns.functions) do
                table.insert(sections, "- `" .. patterns.functions[i] .. "()`")
            end
        end
        
        if #patterns.variables > 0 then
            table.insert(sections, "### Variable Naming:")
            for i = 1, math.min(10, #patterns.variables) do
                table.insert(sections, "- `" .. patterns.variables[i] .. "`")
            end
        end
        table.insert(sections, "")
    end
    
    -- 6. Tags (CollectionService)
    local CollectionService = game:GetService("CollectionService")
    local tags = CollectionService:GetAllTags()
    if #tags > 0 then
        table.insert(sections, "## COLLECTION SERVICE TAGS")
        for _, tag in ipairs(tags) do
            local tagged = CollectionService:GetTagged(tag)
            table.insert(sections, string.format("- `%s` (%d objects)", tag, #tagged))
        end
        table.insert(sections, "")
    end
    
    -- 7. Instructions for AI
    table.insert(sections, "---")
    table.insert(sections, "**INSTRUCTIONS FOR CODE GENERATION:**")
    table.insert(sections, "1. Use EXACT object names from the hierarchy above")
    table.insert(sections, "2. Reference actual RemoteEvents/Functions that exist")
    table.insert(sections, "3. Follow the variable naming patterns shown")
    table.insert(sections, "4. Match the coding style from existing functions")
    table.insert(sections, "5. Use GetFullName() style paths like shown above")
    table.insert(sections, "")
    table.insert(sections, "---")
    table.insert(sections, "")
    
    return table.concat(sections, "\n")
end

function Network.makeRequest(url, method, data)
    -- Check if we're running in a plugin context
    local isPlugin = plugin ~= nil
    if not isPlugin then
        warn("⚠️ NOT RUNNING AS PLUGIN! This script must be installed as a Roblox Studio Plugin.")
        return nil, "Plugin context required"
    end
    
    -- Check if we're in Studio (not in test mode)
    local runService = game:GetService("RunService")
    if not runService:IsStudio() then
        warn("⚠️ Not running in Studio! Please use this plugin in Edit Mode, not in a game.")
        return nil, "Studio edit mode required"
    end
    
    if runService:IsRunning() then
        warn("⚠️ Cannot make HTTP requests in Test Mode!")
        warn("⚠️ Please STOP the test (press Stop button) and use the plugin in Edit Mode.")
        return nil, "HTTP unavailable while playtesting"
    end
    
        local http = game:GetService("HttpService")
    local success, response = pcall(function()
        return http:RequestAsync({
            Url = url,
            Method = method,
            Headers = {
                ["Content-Type"] = "application/json"
            },
            Body = data and http:JSONEncode(data) or nil
        })
    end)

    if not success then
        warn("HTTP request failed:", response)
        return nil, response
    end

    if not response.Success then
        local decodedBody = nil
        if response.Body and response.Body ~= "" then
            local decodeOk, bodyResult = pcall(function()
        return http:JSONDecode(response.Body)
    end)
            if decodeOk then
                decodedBody = bodyResult
            else
                decodedBody = response.Body
            end
        end

        return nil, {
            statusCode = response.StatusCode,
            statusMessage = response.StatusMessage,
            body = decodedBody
        }
    end

    if response.Body and response.Body ~= "" then
        local decodeOk, decoded = pcall(function()
            return http:JSONDecode(response.Body)
        end)
        if decodeOk then
            return decoded
        else
            return response.Body
        end
    end

    return {}
end

-- Function to validate API key
function Network.validateApiKey(apiKey)
    local robloxUserId = Network.getRobloxUserId()
    local response = Network.makeRequest(BACKEND_URL .. "/api/validate-key", "POST", {
        api_key = apiKey,
        roblox_user_id = robloxUserId  -- Send user ID for device binding
    })
    
    return response and response.valid
end

-- Model selection state (shared near top of file)

-- Function to generate code
function Network.generateCode(apiKey, prompt, model)
    -- AGENT-LIKE: Build context dynamically based on user's prompt
    local enhancedPrompt = prompt
    
    if #scriptCache > 0 then
        -- Build intelligent context using the user's actual request
        local contextString = snapshotState:buildGameContext(prompt)
        enhancedPrompt = contextString .. "\n\n**USER REQUEST:**\n" .. prompt
    end
    
    local robloxUserId = Network.getRobloxUserId()
    local response, httpError = Network.makeRequest(BACKEND_URL .. "/api/generate-code", "POST", {
        api_key = apiKey,
        prompt = enhancedPrompt,  -- Send enhanced prompt with agent-searched context
        model = model,
        roblox_user_id = robloxUserId  -- Send user ID for device binding
    })
    
    if response and response.success then
        return response.code
    else
        local errorMessage = "Unknown error"

        if response then
            errorMessage = response.error or response.detail or errorMessage
        elseif httpError then
            if typeof(httpError) == "table" then
                local body = httpError.body
                if typeof(body) == "table" then
                    errorMessage = body.error or body.detail or errorMessage
                elseif typeof(body) == "string" and body ~= "" then
                    errorMessage = body
                elseif httpError.statusCode then
                    errorMessage = string.format("Request failed (HTTP %s)", tostring(httpError.statusCode))
                end
            else
                errorMessage = tostring(httpError)
            end
        end

        return nil, errorMessage
    end
end

function Network.requestAgentResponse(apiKey, payload)
    print("[Electrode] Network.requestAgentResponse called")
    print("[Electrode] Backend URL:", BACKEND_URL .. "/api/agent/chat")
    print("[Electrode] Backend will now retrieve context blocks and process request")
    return Network.makeRequest(BACKEND_URL .. "/api/agent/chat", "POST", payload)
end

function Network.requestAgentContinue(apiKey, continueToken)
    local userId = Network.getRobloxUserId()
    return Network.makeRequest(BACKEND_URL .. "/api/agent/continue", "POST", {
        api_key = apiKey,
        continue_token = continueToken,
        user_id = userId,
        roblox_user_id = userId
    })
end

-- Attach to GeneratorUI to avoid module-level locals
-- API key section is always visible in Settings tab - no need for hide/show functions

-- Attach theme helpers to Style table to avoid module-level local
do
    Style.updateThemeColors = function()
        local palette = getActiveTheme()
        local textPrimary = Color3.fromRGB(235, 240, 255)
        local textSecondary = Color3.fromRGB(190, 198, 230)
        if palette and palette.text then
            textPrimary = palette.text:Lerp(textPrimary, 0.5)
            textSecondary = (palette.subtext or palette.text):Lerp(textSecondary, 0.5)
        end
        return textPrimary, textSecondary
    end

    Style.updateThemeLabels = function(textPrimary, textSecondary)
        Title.TextColor3 = textPrimary
        AgentUI.Header.TextColor3 = textPrimary
        AgentUI.ModelSelectorLabel.TextColor3 = textSecondary
        AgentUI.TrustLabel.TextColor3 = textSecondary
        -- Settings API key label removed
        if AgentUI.SnapshotHeader then
            AgentUI.SnapshotHeader.TextColor3 = textPrimary
        end
        AgentUI.SnapshotTimestampLabel.TextColor3 = textSecondary
        AgentUI.SnapshotSummaryLabel.TextColor3 = textSecondary
        if AgentUI.DiffHeader then
            AgentUI.DiffHeader.TextColor3 = textPrimary
        end
        if AgentUI.PlanHeader then
            AgentUI.PlanHeader.TextColor3 = textPrimary
        end
    end

    Style.updateThemeInputs = function(textPrimary, textSecondary)
        local placeholderColor = currentTheme == "light"
            and Color3.fromRGB(140, 150, 190)
            or Color3.fromRGB(160, 170, 210)
        -- Settings API key input removed
        AgentUI.ChatInputBox.TextColor3 = Color3.fromRGB(255, 255, 255)
        AgentUI.ChatInputBox.PlaceholderColor3 = placeholderColor
    end

    Style.updateThemeTips = function(textSecondary)
        -- Generator prompt tips removed
    end

    Style.updateThemeButtons = function()
        -- Generator buttons removed
        if AgentUI.RefreshContextButton and AgentUI.RefreshContextButton.Text == "Refresh Snapshot" then
            setRefreshButtonIdle()
        end
    end
end

-- Attach updateTheme to Style table to avoid module-level local
Style.updateTheme = function()
    local textPrimary, textSecondary = Style.updateThemeColors()
    Style.updateThemeLabels(textPrimary, textSecondary)
    Style.updateThemeInputs(textPrimary, textSecondary)
    Style.updateThemeTips(textSecondary)
    updateTabVisuals()
    setAgentModel(agentState.model, true)
    Style.setTrustedToggleAppearance(AgentUI.TrustToggle, agentState.trusted)
    -- Generator settings removed
    Style.updateThemeButtons()
end
-- Removed unnecessary local alias to reduce register usage

-- Function to update model display
function agentState:toggleSettingsDropdown()
    if self.settingsFrame then
        self.settingsFrame:Destroy()
        self.settingsFrame = nil
        self.settingsOpen = false
        return
    end
    
    self.settingsOpen = true
    
    -- Create settings frame
    local settingsFrame = Instance.new("Frame")
    settingsFrame.Size = UDim2.new(0.8, 0, 0, 120)
    settingsFrame.Position = UDim2.new(0.1, 0, 0, 310)
    settingsFrame.BackgroundColor3 = Color3.fromRGB(248, 249, 250)
    settingsFrame.BorderColor3 = Color3.fromRGB(233, 236, 239)
    settingsFrame.Parent = MainFrame
    self.settingsFrame = settingsFrame
    
    -- Theme section
    local themeLabel = Instance.new("TextLabel")
    themeLabel.Size = UDim2.new(1, -20, 0, 20)
    themeLabel.Position = UDim2.new(0, 10, 0, 10)
    themeLabel.BackgroundTransparency = 1
    themeLabel.Text = "Theme:"
    themeLabel.TextColor3 = Color3.fromRGB(0, 0, 0)
    themeLabel.TextScaled = true
    themeLabel.Font = Enum.Font.SourceSansBold
    themeLabel.Parent = settingsFrame
    
    local themeButton = Instance.new("TextButton")
    themeButton.Size = UDim2.new(0.4, -5, 0, 25)
    themeButton.Position = UDim2.new(0, 10, 0, 30)
    themeButton.BackgroundColor3 = Color3.fromRGB(102, 126, 234)
    themeButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    themeButton.Text = currentTheme == "light" and "Light" or "Dark"
    themeButton.TextScaled = true
    themeButton.Font = Enum.Font.SourceSansBold
    themeButton.Parent = settingsFrame
    
    -- Model section
    local modelLabel = Instance.new("TextLabel")
    modelLabel.Size = UDim2.new(1, -20, 0, 20)
    modelLabel.Position = UDim2.new(0, 10, 0, 60)
    modelLabel.BackgroundTransparency = 1
    modelLabel.Text = "AI Model:"
    modelLabel.TextColor3 = Color3.fromRGB(0, 0, 0)
    modelLabel.TextScaled = true
    modelLabel.Font = Enum.Font.SourceSansBold
    modelLabel.Parent = settingsFrame
    
    local modelButton = Instance.new("TextButton")
    modelButton.Size = UDim2.new(0.4, -5, 0, 25)
    modelButton.Position = UDim2.new(0, 10, 0, 80)
    modelButton.TextColor3 = Color3.fromRGB(255, 255, 255)
    modelButton.Text = currentModel == "claude" and "Claude" or "GPT-4"
    modelButton.TextScaled = true
    modelButton.Font = Enum.Font.GothamSemibold
    modelButton.Parent = settingsFrame
    Style.styleSecondaryButton(modelButton)

    local function refreshSettingsModelButton()
        if currentModel == "claude" then
            Style.setButtonGradient(
                modelButton,
                Color3.fromRGB(255, 149, 83),
                Color3.fromRGB(255, 94, 58),
                NumberSequence.new({
                    NumberSequenceKeypoint.new(0, 0.1),
                    NumberSequenceKeypoint.new(1, 0.3)
                })
            )
        else
            Style.setButtonGradient(
                modelButton,
                Color3.fromRGB(178, 148, 255),
                Color3.fromRGB(96, 128, 255),
                NumberSequence.new({
                    NumberSequenceKeypoint.new(0, 0.08),
                    NumberSequenceKeypoint.new(1, 0.24)
                })
            )
        end
    end

    refreshSettingsModelButton()
    
    -- Theme button click
    themeButton.MouseButton1Click:Connect(function()
        currentTheme = currentTheme == "light" and "dark" or "light"
        themeButton.Text = currentTheme == "light" and "Light" or "Dark"
        Style.updateTheme()
        
        -- Save theme preference
        pcall(function()
            plugin:SetSetting("Theme", currentTheme)
        end)
        
        -- Theme change notification removed (Generator UI removed)
    end)
    
    -- Model button click
    modelButton.MouseButton1Click:Connect(function()
        currentModel = currentModel == "gpt" and "claude" or "gpt"
        modelButton.Text = currentModel == "claude" and "Claude" or "GPT-4"
        refreshSettingsModelButton()
        
        -- Save model preference
        pcall(function()
            plugin:SetSetting("SelectedModel", currentModel)
        end)
        
        -- Model change notification removed (Generator UI removed)
    end)
end

-- API key handlers removed - API key entry moved to popup only

-- Generator Sync Context button removed - Agent handles context sync automatically

-- Generator Settings button removed

-- Generator button handler removed - Generator functionality removed

-- Plugin button click handler
PluginButton.Click:Connect(function()
    Widget.Enabled = not Widget.Enabled
end)

-- Load saved API key if available
function loadSavedApiKey()
    local success, savedKey = pcall(function()
        return plugin:GetSetting("ApiKey")
    end)
    
    if success and savedKey and savedKey ~= "" then
        -- Settings API key input removed - using popup only
        cachedApiKey = savedKey
        
        -- Only validate API key if not in test mode (HTTP doesn't work in test mode)
        if not RunService:IsRunning() then
            if Network.validateApiKey(savedKey) then
                -- API key is valid, no popup needed
            else
                -- Show API key popup if invalid (but wait a bit so UI is ready)
                task.wait(0.5)
                showApiKeyPopup()
            end
        end
    else
        -- No API key saved, show popup (but wait a bit so UI is ready)
        task.wait(0.5)
        showApiKeyPopup()
    end
end

-- Load saved settings if available
function loadSavedSettings()
    -- Settings loading removed - no Generator settings to load
    -- Future settings can be loaded here
end

-- API key auto-save already handled above

-- Attach error detection to errorState table to avoid module-level local
do
    errorState.isRealError = function(message, messageType)
        local msgLower = string.lower(message or "")
        
        -- Direct error types - always errors
        if messageType == Enum.MessageType.MessageError then
            return true
        end
        
        -- Warnings - mostly errors except deprecation noise
        if messageType == Enum.MessageType.MessageWarning then
            if msgLower:find("deprecated") or msgLower:find("deprecate") then
                return false
            end
            return true
        end
        
        -- Detect actual runtime/logic errors printed as MessageOutput
        if messageType == Enum.MessageType.MessageOutput then
            return (
                msgLower:find("attempt to") or
                msgLower:find("is not a valid member") or
                msgLower:find("expected") or
                msgLower:find("missing") or
                msgLower:find("failed") or
                msgLower:find("syntax") or
                msgLower:find("unknown") or
                msgLower:find("unable to") or
                msgLower:find("cannot") or
                msgLower:find("invalid") or
                msgLower:find("stack traceback")
            )
        end
        
        return false
    end

    errorState.getRecentErrors = function(limit)
        -- Return the accumulated errors from real-time event listening
        -- Deduplicate and limit results
        local out = {}
        local seen = {}
        local currentTime = tick()
        local timeLimit = 600 -- 10 minutes in seconds
        
        -- Iterate through accumulated errors (most recent first since they're added at the end)
        for i = #errorState.errors, 1, -1 do
            local err = errorState.errors[i]
            if err and err.message then
                -- Skip errors older than 10 minutes
                local errTime = err.timestamp or currentTime
                if currentTime - errTime <= timeLimit then
                    -- Deduplicate
                    if not seen[err.message] then
                        seen[err.message] = true
                        table.insert(out, err)
                        if #out >= (limit or 50) then
                            break
                        end
                    end
                end
            end
        end
        
        return out
    end

    errorState.cleanupOldErrors = function()
        -- Remove errors older than 10 minutes from errorState.errors
        local currentTime = tick()
        local timeLimit = 600 -- 10 minutes in seconds
        
        for i = #errorState.errors, 1, -1 do
            local err = errorState.errors[i]
            if err and err.timestamp then
                if currentTime - err.timestamp > timeLimit then
                    table.remove(errorState.errors, i)
                end
            else
                -- Remove invalid error entry
                table.remove(errorState.errors, i)
            end
        end
    end

    errorState.updateErrorDisplay = function()
        -- Clean up old errors first
        errorState.cleanupOldErrors()
        
        -- Get recent errors (deduplicated, limited)
        local recentErrors = errorState.getRecentErrors(50)
        local errorCount = #recentErrors
        
        -- Update UI based on playtesting state
        if errorState.isPlaytesting then
            -- During playtesting: show real-time monitor card
            if not ErrorMonitorCard.Visible then
                ErrorMonitorCard.Visible = true
            end
            if errorCount > 0 then
                AgentUI.ErrorCountLabel.Text = string.format("%d error%s found", errorCount, errorCount > 1 and "s" or "")
                AgentUI.ErrorDebugButton.Visible = true
            else
                AgentUI.ErrorCountLabel.Text = "0 errors found"
                AgentUI.ErrorDebugButton.Visible = false
            end
        else
            -- After playtesting: hide monitor card, show fix button if errors exist
            if ErrorMonitorCard.Visible then
                ErrorMonitorCard.Visible = false
            end
            if errorCount > 0 then
                AgentUI.ErrorFixButton.Visible = true
                AgentUI.ErrorFixButton.Text = string.format("Fix %d error%s?", errorCount, errorCount > 1 and "s" or "")
            else
                AgentUI.ErrorFixButton.Visible = false
            end
        end
    end

    errorState.startMonitoring = function()
        -- Clear previous errors when starting new playtest
        errorState.errors = {}
        errorState.isPlaytesting = true
        -- Show real-time error monitor
        ErrorMonitorCard.Visible = true
        AgentUI.ErrorCountLabel.Text = "0 errors found"
        AgentUI.ErrorDebugButton.Visible = false
        
        -- Connect to LogService.MessageOut for real-time error detection
        if errorState.logServiceConnection then
            errorState.logServiceConnection:Disconnect()
            errorState.logServiceConnection = nil
        end
        
        -- Try connecting to MessageOut event for real-time error detection
        -- The event signature may vary, so we handle multiple formats
        local function handleLogMessage(messageOutput, messageTypeParam)
            local message = ""
            local messageType = nil
            local timestamp = tick()
            
            -- Handle different event formats
            if type(messageOutput) == "table" then
                -- Table format: {Message = "...", MessageType = Enum.MessageType, Timestamp = number}
                message = messageOutput.Message or ""
                messageType = messageOutput.MessageType
                timestamp = messageOutput.Timestamp or timestamp
            elseif type(messageOutput) == "string" then
                -- String format: message is first param, messageType is second param
                message = messageOutput
                messageType = messageTypeParam
            end
            
            -- Check if it's a real error
            if messageType and errorState.isRealError(message, messageType) then
                -- Deduplicate: check if we've seen this exact message recently
                local seen = false
                local currentTime = timestamp
                for i, err in ipairs(errorState.errors) do
                    if err.message == message then
                        -- Update timestamp if we see it again (more recent)
                        err.timestamp = currentTime
                        seen = true
                        break
                    end
                end
                
                if not seen and message ~= "" then
                    table.insert(errorState.errors, {
                        message = message,
                        timestamp = currentTime,
                        messageType = messageType
                    })
                    -- Update UI immediately when new error is detected
                    errorState.updateErrorDisplay()
                end
            end
        end
        
        -- Try to connect to LogService.MessageOut event
        local eventConnected = false
        if LogService.MessageOut then
            local success, err = pcall(function()
                errorState.logServiceConnection = LogService.MessageOut:Connect(handleLogMessage)
                eventConnected = true
            end)
            
            if not success then
                warn("[Electrode] Failed to connect to LogService.MessageOut: " .. tostring(err))
            end
        end
        
        -- If event connection failed, we'll use periodic polling of GetLogHistory as fallback
        errorState.useEventBasedDetection = eventConnected
        
        -- Also update UI periodically and poll for errors if event-based detection failed
        if errorState.monitorConnection then
            errorState.monitorConnection:Disconnect()
        end
        errorState.monitorConnection = RunService.Heartbeat:Connect(function()
            -- Update display every ~1 second to clean up old errors
            if not errorState.lastErrorCheckTime then
                errorState.lastErrorCheckTime = tick()
            end
            local currentTime = tick()
            if currentTime - errorState.lastErrorCheckTime >= 1.0 then
                errorState.lastErrorCheckTime = currentTime
                
                -- If event-based detection failed, poll GetLogHistory as fallback
                if not errorState.useEventBasedDetection then
                    -- Poll GetLogHistory to find errors
                    local success, logs = pcall(function()
                        return LogService:GetLogHistory()
                    end)
                    
                    if success and logs then
                        local currentTimeForLogs = tick()
                        local seen = {}
                        
                        -- Check recent logs (last 50 entries)
                        local startIdx = math.max(1, #logs - 49)
                        for i = #logs, startIdx, -1 do
                            local logEntry = logs[i]
                            if logEntry then
                                local message = logEntry.message or logEntry.Message or ""
                                local messageType = logEntry.messageType or logEntry.MessageType
                                local timestamp = logEntry.timestamp or logEntry.Timestamp or currentTimeForLogs
                                
                                if messageType and errorState.isRealError(message, messageType) then
                                    -- Check if we already have this error
                                    if not seen[message] then
                                        seen[message] = true
                                        local alreadyHave = false
                                        for j, err in ipairs(errorState.errors) do
                                            if err.message == message then
                                                alreadyHave = true
                                                -- Update timestamp
                                                err.timestamp = timestamp
                                                break
                                            end
                                        end
                                        
                                        if not alreadyHave and message ~= "" then
                                            table.insert(errorState.errors, {
                                                message = message,
                                                timestamp = timestamp,
                                                messageType = messageType
                                            })
                                        end
                                    end
                                end
                            end
                        end
                    end
                end
                
                errorState.updateErrorDisplay()
            end
        end)
    end

    errorState.stopMonitoring = function()
        if not errorState.isPlaytesting then
            return
        end
        
        errorState.isPlaytesting = false
        
        -- Don't disconnect logServiceConnection - keep listening for errors even after playtesting stops
        -- Don't disconnect monitor connection - keep updating UI periodically
        
        -- Update display immediately when playtesting stops
        errorState.updateErrorDisplay()
        
        -- Check if there's a pending error request from test mode
        if errorState.pendingErrorRequest and #errorState.pendingErrorMessages > 0 then
            -- Send the queued error request now that test mode has stopped
            task.delay(0.5, function()
                local errorOutput = {}
                for i, msg in ipairs(errorState.pendingErrorMessages) do
                    table.insert(errorOutput, string.format("%d. %s", i, msg))
                end
                local prompt = string.format("Errors have been found when playtesting. Please concisely fix. Here is the output:\n\n%s", table.concat(errorOutput, "\n"))
                errorState.pendingErrorRequest = false
                errorState.pendingErrorMessages = {}
                AgentUI.ChatInputBox.Text = prompt
                sendAgentMessage()
            end)
        end
    end

    errorState.clearErrors = function()
        errorState.errors = {}
        AgentUI.ErrorFixButton.Visible = false
        if ErrorMonitorCard then
            ErrorMonitorCard.Visible = false
            AgentUI.ErrorCountLabel.Text = "0 errors found"
            AgentUI.ErrorDebugButton.Visible = false
        end
    end

    errorState.sendToAI = function()
        if #errorState.errors == 0 then
            return
        end
        
        -- Check if we're in test mode - HTTP requests don't work during playtesting
        if RunService:IsRunning() then
            -- Queue the request to be sent when test mode stops
            errorState.pendingErrorRequest = true
            errorState.pendingErrorMessages = {}
            for i, err in ipairs(errorState.errors) do
                table.insert(errorState.pendingErrorMessages, err.message)
            end
            -- Show message that request will be sent when test stops
            appendConversationMessage("assistant", "⚠️ Error debugging request queued. It will be sent automatically when you stop the test.", nil, true)
            -- Don't clear errors yet - keep them visible until test stops
            return
        end
        
        -- Build simple error output (just the messages)
        local errorOutput = {}
        for i, err in ipairs(errorState.errors) do
            table.insert(errorOutput, string.format("%d. %s", i, err.message))
        end
        
        -- Simple prompt - minimal context
        local prompt = string.format("Errors have been found when playtesting. Please concisely fix. Here is the output:\n\n%s", table.concat(errorOutput, "\n"))
        
        -- Clear errors and hide button
        errorState.clearErrors()
        
        -- Send to AI
        AgentUI.ChatInputBox.Text = prompt
        sendAgentMessage()
    end
end

-- Connect error fix button (when playtest stops)
AgentUI.ErrorFixButton.MouseButton1Click:Connect(function()
    errorState.sendToAI()
end)

-- Connect real-time error debug button (during playtesting)
AgentUI.ErrorDebugButton.MouseButton1Click:Connect(function()
    errorState.sendToAI()
end)

-- Monitor playtest state and continuously check for errors
errorState.lastPlaytestState = RunService:IsRunning()
-- Start continuous error monitoring immediately (works even without playtesting)
errorState.startMonitoring()

RunService.Heartbeat:Connect(function()
    local isRunning = RunService:IsRunning()
    if isRunning ~= errorState.lastPlaytestState then
        errorState.lastPlaytestState = isRunning
        if isRunning then
            -- Playtesting started - ensure monitoring is active
            if not errorState.isPlaytesting then
                errorState.startMonitoring()
            end
        else
            -- Playtesting stopped - update state but keep monitoring
            errorState.stopMonitoring()
        end
    end
end)

-- ============================================================================
-- QUICK ACTIONS MENU - Create actions once and reuse to avoid duplicate ID errors
-- ============================================================================
_G.QuickActionsMenu = plugin:CreatePluginMenu("ElectrodeQuickActions", "Electrode Quick Actions")
_G.QuickActionsHandler = function(actionId)
    local s = SelectionService:Get()
    local o = s and s[1]
    if getActiveApiKey() == "" then
        if actionId == "NeedApiKey" then Widget.Enabled = true activeTab = "Agent" updateTabVisuals() showApiKeyPopup() end
        return
    end
    Widget.Enabled = true
    if activeTab ~= "Agent" then activeTab = "Agent" updateTabVisuals() end
    task.wait(0.1)
    if o and AgentUI and AgentUI.ChatInputBox then
        local p, c = o:GetFullName(), o.ClassName
        if actionId == "ExplainScript" and o:IsA("LuaSourceContainer") then
            AgentUI.ChatInputBox.Text = string.format("Explain this script step by step. Show what it does, how it works, and any important details:\n\nScript: %s", p)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        elseif actionId == "FixErrors" and o:IsA("LuaSourceContainer") then
            AgentUI.ChatInputBox.Text = string.format("Fix any errors in this script. Analyze it thoroughly and correct all issues:\n\nScript: %s", p)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        elseif actionId == "RefactorScript" and o:IsA("LuaSourceContainer") then
            AgentUI.ChatInputBox.Text = string.format("Refactor this script to improve code quality, organization, and maintainability:\n\nScript: %s", p)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        elseif actionId == "OptimizeScript" and o:IsA("LuaSourceContainer") then
            AgentUI.ChatInputBox.Text = string.format("Optimize this script for better performance. Identify bottlenecks and improve efficiency:\n\nScript: %s", p)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        elseif actionId == "DocumentScript" and o:IsA("LuaSourceContainer") then
            AgentUI.ChatInputBox.Text = string.format("Add comprehensive documentation to this script. Include comments explaining what it does:\n\nScript: %s", p)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        elseif actionId == "GenerateScriptFor" then
            AgentUI.ChatInputBox.Text = string.format("Generate a script for: %s (%s)\n\nWhat should this script do?", p, c)
            AgentUI.ChatInputBox:CaptureFocus()
        elseif actionId == "AddAIBehavior" and (o:IsA("BasePart") or o:IsA("Model")) then
            AgentUI.ChatInputBox.Text = string.format("Add AI-powered behavior to: %s\n\nWhat behavior should this have? (e.g., 'make it glow when touched', 'create a proximity prompt', 'add NPC behavior')", p)
            AgentUI.ChatInputBox:CaptureFocus()
        elseif actionId == "GenerateSystem" and o:IsA("Folder") then
            AgentUI.ChatInputBox.Text = string.format("Generate a complete system inside the folder: %s\n\nWhat system should I create? (e.g., 'inventory system', 'shop system', 'quest system')", p)
            AgentUI.ChatInputBox:CaptureFocus()
        elseif actionId == "ExplainInstance" then
            AgentUI.ChatInputBox.Text = string.format("Explain what this instance does and its purpose:\n\nInstance: %s (%s)", p, c)
            AgentUI.ChatInputBox:CaptureFocus()
            task.wait(0.15)
            sendAgentMessage()
        end
    end
    if actionId == "GenerateScript" and AgentUI and AgentUI.ChatInputBox then
        AgentUI.ChatInputBox.Text = "Generate a new script. What should it do and where should it go?"
        AgentUI.ChatInputBox:CaptureFocus()
    end
end
-- Create all actions once at initialization (stored in globals to avoid locals)
_G.QuickActions = {}
_G.QuickActions.NeedApiKey = plugin:CreatePluginAction("QANeedApiKey", "API Key Required", "")
_G.QuickActions.NeedApiKey.Triggered:Connect(function() _G.QuickActionsHandler("NeedApiKey") end)
_G.QuickActions.ExplainScript = plugin:CreatePluginAction("QAExplainScript", "Explain This Script", "")
_G.QuickActions.ExplainScript.Triggered:Connect(function() _G.QuickActionsHandler("ExplainScript") end)
_G.QuickActions.FixErrors = plugin:CreatePluginAction("QAFixErrors", "Fix Errors in This", "")
_G.QuickActions.FixErrors.Triggered:Connect(function() _G.QuickActionsHandler("FixErrors") end)
_G.QuickActions.RefactorScript = plugin:CreatePluginAction("QARefactorScript", "Refactor This Script", "")
_G.QuickActions.RefactorScript.Triggered:Connect(function() _G.QuickActionsHandler("RefactorScript") end)
_G.QuickActions.OptimizeScript = plugin:CreatePluginAction("QAOptimizeScript", "Optimize This Script", "")
_G.QuickActions.OptimizeScript.Triggered:Connect(function() _G.QuickActionsHandler("OptimizeScript") end)
_G.QuickActions.DocumentScript = plugin:CreatePluginAction("QADocumentScript", "Document This Script", "")
_G.QuickActions.DocumentScript.Triggered:Connect(function() _G.QuickActionsHandler("DocumentScript") end)
_G.QuickActions.GenerateScriptFor = plugin:CreatePluginAction("QAGenerateScriptFor", "Generate Script", "")
_G.QuickActions.GenerateScriptFor.Triggered:Connect(function() _G.QuickActionsHandler("GenerateScriptFor") end)
_G.QuickActions.AddAIBehavior = plugin:CreatePluginAction("QAAddAIBehavior", "Add AI Behavior", "")
_G.QuickActions.AddAIBehavior.Triggered:Connect(function() _G.QuickActionsHandler("AddAIBehavior") end)
_G.QuickActions.GenerateSystem = plugin:CreatePluginAction("QAGenerateSystem", "Generate System", "")
_G.QuickActions.GenerateSystem.Triggered:Connect(function() _G.QuickActionsHandler("GenerateSystem") end)
_G.QuickActions.ExplainInstance = plugin:CreatePluginAction("QAExplainInstance", "Explain Instance", "")
_G.QuickActions.ExplainInstance.Triggered:Connect(function() _G.QuickActionsHandler("ExplainInstance") end)
_G.QuickActions.GenerateScript = plugin:CreatePluginAction("QAGenerateScript", "Generate New Script", "")
_G.QuickActions.GenerateScript.Triggered:Connect(function() _G.QuickActionsHandler("GenerateScript") end)
_G.QuickActions.OpenAgent = plugin:CreatePluginAction("QAOpenAgent", "Open Agent Chat", "")
_G.QuickActions.OpenAgent.Triggered:Connect(function() Widget.Enabled = true activeTab = "Agent" updateTabVisuals() end)
_G.updateQuickActionsMenu = function()
    local s = SelectionService:Get()
    local o = s and s[1]
    _G.QuickActionsMenu:Clear()
    if getActiveApiKey() == "" then
        _G.QuickActionsMenu:AddAction(_G.QuickActions.NeedApiKey)
        return
    end
    if o then
        if o:IsA("LuaSourceContainer") then
            _G.QuickActionsMenu:AddAction(_G.QuickActions.ExplainScript)
            _G.QuickActionsMenu:AddAction(_G.QuickActions.FixErrors)
            _G.QuickActionsMenu:AddAction(_G.QuickActions.RefactorScript)
            _G.QuickActionsMenu:AddAction(_G.QuickActions.OptimizeScript)
            _G.QuickActionsMenu:AddSeparator()
            _G.QuickActionsMenu:AddAction(_G.QuickActions.DocumentScript)
        elseif o:IsA("BasePart") or o:IsA("Model") then
            _G.QuickActionsMenu:AddAction(_G.QuickActions.GenerateScriptFor)
            _G.QuickActionsMenu:AddAction(_G.QuickActions.AddAIBehavior)
        elseif o:IsA("Folder") then
            _G.QuickActionsMenu:AddAction(_G.QuickActions.GenerateSystem)
        else
            _G.QuickActionsMenu:AddAction(_G.QuickActions.ExplainInstance)
            _G.QuickActionsMenu:AddAction(_G.QuickActions.GenerateScriptFor)
        end
        _G.QuickActionsMenu:AddSeparator()
    end
    _G.QuickActionsMenu:AddAction(_G.QuickActions.GenerateScript)
    _G.QuickActionsMenu:AddAction(_G.QuickActions.OpenAgent)
end
SelectionService.SelectionChanged:Connect(_G.updateQuickActionsMenu)
task.defer(_G.updateQuickActionsMenu)

-- ============================================================================

-- Load saved settings on startup
loadSavedApiKey()
loadSavedSettings()
loadAgentSession()
updateVFXTabVisibility()
task.defer(function()
    captureSnapshot("startup")
end)

print("electrode initialized")
print("electrode initialized")

-- Legacy global function aliases for compatibility with precompiled bundle
_G.getActiveApiKey = getActiveApiKey
_G.sendAgentMessage = sendAgentMessage
_G.requestAgentResponse = Network.requestAgentResponse
_G.requestAgentContinue = Network.requestAgentContinue
_G.serializeConversation = serializeConversation
_G.runAgentActions = runAgentActions
_G.updatePlanUI = updatePlanUI
_G.clearScrollingFrame = clearScrollingFrame
_G.captureSnapshot = captureSnapshot
_G.saveAgentSession = saveAgentSession


